/*
* Copyright (C) The Community OpenORB Project. All rights reserved.
*
* This software is published under the terms of The OpenORB Community Software
* License version 1.0, a copy of which has been included with this distribution
* in the LICENSE.txt file.
*/
package org.openorb.util;

import org.omg.CORBA.SystemException;
import org.omg.CORBA.UserException;

/**
 * A utility class for optionaly attaching causal objects to exceptions
 * if the exception implementation supports <code>initCause</code>
 *
 * @author Richard G Clark
 */
public final class ExceptionTool 
{
    private static final boolean JRE_1_4 = JREVersion.V1_4;
    private static final boolean INIT_CAUSE_SUPPORTED;
    
    static 
    {
        final Class[] parameterTypes = { Throwable.class };
        
        boolean supported = false;
        try 
        {
            SystemException.class.getMethod( "initCause", parameterTypes );
            UserException.class.getMethod( "initCause", parameterTypes );
            supported = true;
        } 
        catch ( final NoSuchMethodException e2 )
        {
            // now we know that the JDK is older than 1.4
        }

        INIT_CAUSE_SUPPORTED = supported || JRE_1_4;
    }
    
    private ExceptionTool()
    {
    }
    
    /**
     * Attaches the cause to the exception if this operation is supported
     * by the current environment.
     *
     * @param e the exception to attach the cause to
     * @param cause the cause of this specified exception
     *
     * @return the parameter e
     */
    public static SystemException initCause( final SystemException e,
                                             final Throwable cause ) 
    {
        if ( INIT_CAUSE_SUPPORTED )
        {
            e.initCause( cause );
        }
        return e;
    }

    /**
     * Attaches the cause to the exception if this operation is supported
     * by the current environment.
     *
     * @param e the exception to attach the cause to
     * @param cause the cause of this specified exception
     *
     * @return the parameter e
     */
    public static UserException initCause( final UserException e,
                                           final Throwable cause ) 
    {
        if ( INIT_CAUSE_SUPPORTED )
        {
            e.initCause( cause );
        }
        return e;
    }
    
    /**
     * Attaches the cause to the exception if this operation is supported
     * by the current environment.
     *
     * @param e the exception to attach the cause to
     * @param cause the cause of this specified exception
     *
     * @return the parameter e
     */
    public static Throwable initCause( final Throwable e,
                                       final Throwable cause )
    {
        if ( JRE_1_4 )
        {
            e.initCause( cause );
        }
        return e;
    }
    
    /**
     * Attaches the cause to the exception if this operation is supported
     * by the current environment.
     *
     * @param e the exception to attach the cause to
     * @param cause the cause of this specified exception
     *
     * @return the parameter e
     */
    public static RuntimeException initCause( final RuntimeException e,
                                              final Throwable cause )
    {
        if ( JRE_1_4 )
        {
            e.initCause( cause );
        }
        return e;
    }

    /**
     * Attaches the cause to the exception if this operation is supported
     * by the current environment.
     *
     * @param e the exception to attach the cause to
     * @param cause the cause of this specified exception
     *
     * @return the parameter e
     */
    public static Error initCause( final Error e, final Throwable cause )
    {
        if ( JRE_1_4 )
        {
            e.initCause( cause );
        }
        return e;
    }
}

