
/* This file is part of xtmodcoll-3.0, (c) xtmod.com
 * 
 * xtmodcoll-3.0 is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * xtmodcoll-3.0 is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser Public License
 * along with xtmodcoll-3.0.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.xtmod.util.algo;

import java.lang.reflect.Array;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Vector;



public class ListAnalysis<E> {

	private List<E> objects;
	private Feature<E, ?> feat;
	

	public ListAnalysis(List<E> objects) {
		this.objects = objects;
		this.feat = new Feature<E, String>(){
			public String feature (E x) {
				return String.valueOf(x);
			}
		};
	}

	public ListAnalysis(List<E> objects, Feature<E, ?> feat) {
		this.objects = objects;
		this.feat = feat;
	}
	

	
	// =====================================================================================
	// Analysis

	
	/**
	 * 
	 * @param array
	 * @return
	 */
	public Integer[] equivalenceClasses () {
		if (objects.size() == 0)
			return new Integer[]{};
		return equivalenceClasses(0, objects.size()-1);
	}

	/**
	 * 
	 * @param array
	 * @param from
	 * @param to
	 * @param feat
	 * @return
	 */
	public Integer[] equivalenceClasses (int from, int to) {
		checkIndices(from, to);

		Vector<Object> temp = new Vector<Object>(objects.size());

		// --- first ---
		Object proto = feat.feature(objects.get(from));
		temp.add(from);

		// --- remainder ---
		for (int j = from + 1; j <= to; j++) {
			Object f = feat.feature(objects.get(j));
			
			if (! proto.equals(f)) {
				proto = f;
				temp.add(j);
			}
		}

		Integer[] ret = new Integer[temp.size()];
		temp.toArray(ret);
		return ret;
	}
	
	
	

	// =====================================================================================
	// Splitting / Uniquifying

	/**
	 * @return
	 */
	public List<List<E>> split () {
		Integer[] classes = equivalenceClasses();
		
		return split(classes);
	}

	/**
	 * 
	 * @param array
	 * @param from
	 * @param to
	 * @return
	 */
	public List<List<E>> split (int from, int to) {
		Integer[] classes = equivalenceClasses(from, to);
		
		return split(classes);
	}
	
	
	/** 
	 * The actual logic, it relies on the Analysis results.
	 */
	private List<List<E>> split (Integer[] classes) {
		List<List<E>> ret = new ArrayList<List<E>>(); 
		
		for (int i = 0; i < classes.length; i++) {
			int from = classes[i];

			int to;
			if (i < classes.length - 1)
				to = classes[i + 1] - 1;
			else
				to = objects.size() - 1;
			
			List<E> list = new ArrayList<E>(objects.subList(from, to+1));
			ret.add (list);
		}

		return ret;
	}

	
	
	
	/**
	 * Uniquify.
	 * 
	 * @param arr
	 * @return
	 */
	public List<E> uniquify () {
		return uniquify(0, objects.size() - 1);
	}


	/**
	 * Uniquify.
	 * 
	 * @param array
	 * @param from
	 * @param to
	 * @param forwards
	 * @return
	 */
	public List<E> uniquify (int from, int to) {
		Integer[] classes = equivalenceClasses(from, to);

		return uniquify(from, to, classes);
	}

	
	/** 
	 * The actual logic, it relies on the Analysis results.
	 */
	private List<E> uniquify (int from, int to, Integer[] classes) {

		List<E> ret = new ArrayList<E>();

		for (int i = 0; i < classes.length; i++) {
			int idx = classes[i];
			ret.add(objects.get(idx));
		}

		return ret;
	}

	
	
	
	
	// =====================================================================================
	// Join / Reverse
	
	

	/**
	 * Join.
	 * 
	 * This is a static method.
	 * 
	 * @param lists
	 * @return
	 */
	static public <E> List<E> join (List<List<E>> lists) {

		// calculate size of result list
		int totalLength = 0;
		for (List<E> list : lists)
			totalLength += list.size();

		// create and populate result list
		List<E> ret = new ArrayList<E>(totalLength);
		for (List<E> list : lists)
			ret.addAll(list);
		
		return ret;
	}



	/**
	 * Reverse.
	 */
	public List<E> reverse () {
		List<E> ret = new ArrayList<E>(objects);
		Collections.reverse(ret);
		return ret;
	}



	
	// =====================================================================================
	// Non-interesting stuff
	
	
	/**
	 * Check Indices.
	 * 
	 * @param array
	 * @param from
	 * @param to
	 */
	private void checkIndices (int from, int to) {

		if (from < 0)
			throw new ArrayIndexOutOfBoundsException("from=" + from + " < 0");

		if (to >= objects.size())
			throw new ArrayIndexOutOfBoundsException("to=" + to + " > " + (objects.size() - 1));

		if (from > to)
			throw new IllegalArgumentException("from=" + from + " > " + to + "=to");

	}


}
