/******************************************************************************
* E.S.O. - VLT project
*
* "@(#) $Id: lcubootAutoDrv.c,v 1.2 2008/10/27 14:53:57 bjeram Exp $"
*
* who        when       what
* ---------  ---------- ---------------------------------------------------
* S.Sandrock 23.04.1995 New
* S.Sandrock 28.08.1995 SPR 950473: increased default number of dev channels.
*/

/************************************************************************
*   NAME
*	lcubootAutoDrv,
*	lcubootAutoDevRegister,
*	lcubootAutoDrvInstall,
*	lcubootAutoDevCreate,
*	lcubootAutoDevCheck
*	- Automatic LCU Driver and Device Installation Facility
*
*   SYNOPSIS
*	int lcubootAutoDevRegister(const char *devName, 
*				   int addrSpace, void *busAddr,
*				   int lccArg1, ... lccArg3)
*
*	int lcubootAutoDrvInstall(const char *moduleName, 
*				  int numDevices, 
*				  const char *drvFuncName, 
*				  const char *devFuncName, 
*				  int drvArg1, ... drvArg6)
*
*	int lcubootAutoDevCreate(const char *devName, int arg1, ... arg9)
*
*	int lcubootAutoDevCheck(const char *devBaseName, int nominalCount)
*
*   DESCRIPTION
*	These functions support the automatic installation of LCU 
*	drivers and devices. All functions are intended to be
*	used directly from the VxWorks shell in a boot-script.
*
*	The script is aborted when a fatal error condition occurs, which
*	is signalled as a log message in the form:
*
*	<tid> (tShell): lcuboot: <message>: <faulty item>:<errno | faulty item>
*	<tid> (tShell): --- SCRIPT ABORTED ---
*
*	lcubootAutoDevRegister - registers a device for later creation and 
*		registration in LCC. Internal variables are set that are used
*		in subsequent calls of `lcubootAutoDrvInstall' to install the
*		driver and `lcubootAutoDevCreate' to create the devices.
*
*		`busAddr' must be a valid VMEbus address in `addrSpace':
*
*			VMEbus Address Modifier	   addrSpace       busAddr
*			----------------------- --------------- -------------
*			VME_AM_SUP_SHORT_IO             0x2d    0 .. 0xffff
*			VME_AM_STD_SUP_DATA             0x3d    0 .. 0xffffff
*			VME_AM_EXT_SUP_DATA             0x0d    0 .. 0xffffffff
*
*		`lccArg1' ff. are are saved and passed to `lccRegisterDevice' 
*		in a subsequent call of `lcubootAutoLccRegisterDevs' to make 
*		all devices known to LCC, see lccRegisterDevice(3).
*
*		The number of devices found so far is returned. This count is
*		reset to zero by `lcubootAutoDrvInstall'.
*
*		The script is aborted if the internal device table is full,
*		or if the given address-space and bus-address could not be 
*		converted into a valid local address.
*
*	lcubootAutoDrvInstall - conditionally loads and installs a driver 
*		and prepares all data for subsequent device installation.
*
*		`moduleName' is searched in BINPATH and loaded.
*		The rest of the parameters are usually defaulted (see below).
*		No operation is done and OK is returned if the number of 
*		devices evaluates to zero (see `numDevices' below),
*		otherwise the installation is done accordingly.
*
*		`drvFuncName' is the driver install function (e.g "xxxDrv").
*		`devFuncName' is the device install function ("xxxDevCreate").
*		`drvArg1' ff. are passed to the driver install function.
*
*		The script is aborted in the following cases:
*		- the driver module could not be found or loaded.
*		- the driver installation failed, i.e. it did not return OK
*		- the device installation function could not be found
*
*		The following parameters are defaulted - when omitted - with:
*
*		  `numDevices' = effective count from `lcubootAutoDevRegister',
*			or - if this is zero - from the variable 
*			`lcubootProbeCount' as set by `lcubootAutoProbe'.
*			Both counts are reset to zero by the function.
*
*		  `drvFuncName' = "`moduleName'`lcubootAutoDrvFuncSuffix'"
*		  `devFuncName' = "`moduleName'`lcubootAutoDevFuncSuffix'"
*
*		  `drvArg1' = `numDevices' (after resolving as above)
*		  `drvArg2' = `lcubootAutoDrvArg2'
*		  `drvArg3' = `lcubootAutoDrvArg3'
*
*		See VARIABLES for the contents of `lcubootAutoD...'.
*
*		To install the corresponding devices, at least as many
*		consecutive calls of `lcubootAutoDevCreate' must follow
*		as the evaluated number of devices implies.
*
*	lcubootAutoDevCreate - performs the installation of a device using
*		the function previously defined with `lcubootAutoDrvInstall'.
*		The name of the device is given by `devName', which is passed
*		with up to nine further arguments to the install-function.
*
*		No operation is done and OK is returned if the number
*		of devices to be installed as defined before with 
*		`lcubootAutoDrvInstall' has already been reached,
*		or if the corresponding address given with 
*		`lcubootAutoDevRegister' is empty.
*
*		It returns OK is the device installation is successful,
*		otherwise the script is aborted.
*
*		Note that there must be as many calls of this function stated
*		in the boot-script as the maximum number of devices that shall
*		be supported. However, usually fewer devices are actually 
*		installed.
*
*		Note also that the base-addresses given as arguments to this
*		function may be different from the probe-addresses stated
*		in the call of `lcubootAutoDevRegister'.
*
*	lcubootAutoDevCheck - checks whether the number of devices installed
*		in the system are equal to a given value.
*
*		`devBaseName' denotes the name-prefix of the device family,
*		e.g. "/acro" for digital I/O devices.
*		`nominalCount' is the expected number of installed devices, a
*		negative value means no check shall be done and returns OK.
*		OK is returned if the actual and nominal count are equal,
*		otherwise the script is aborted.
*
*		This can be used at the end of the boot-script to verify that
*		the expected number of devices are actually installed.
*		Note however that the boot-script must then be updated 
*		according to the HW configuration.
*
*   VARIABLES
*	The following global variables are provided and can be modified by
*	the user in exceptional cases:
*
*	lcubootAutoDrvFuncSuffix - Default suffix to the module-name to  
*		yield the driver install function.
*		The variable is preset with: "Drv"
*		The maximum length is 15 chars.
*
*	lcubootAutoDevFuncSuffix - Default suffix to the module-name to  
*		yield the device install function.
*		The variable is preset with: "DevCreate"
*		The maximum length is 15 chars.
*
*	lcubootAutoDrvArg2 - Default 2nd parameter for the driver install 
*		function, usually the number of channels.
*		The variable is preset with: 25
*		Note that that this parameter can be explicitly given as
*		argument to `lcubootAutoDrvInstall'.
*
*	lcubootAutoDrvArg3 - Default 3rd parameter for the driver install 
*		function, usually the timeout in ticks. 
*		The variable is preset with: 100
*		Note that that this parameter can be explicitly given as
*		argument to `lcubootAutoDrvInstall'.
*
*   FILES
*
*   ENVIRONMENT
*	BINPATH - colon-separated directory searchpath for binary modules
*
*   COMMANDS
*
*   RETURN VALUES
*
*   CAUTIONS 
*	Most functions are not reentrant and should therefore only be used
*	from LCU boot-scripts, where reentrancy is not important!
*
*	Not more than 64 devices are supported in one LCU.
*
*	The global variable `lcubootAutoDrvArg2' contains the default number
*	of device channels that can be opened to all devices of a driver.
*	When very many devices are present, this might have to be increased.
*
*   EXAMPLES
*	This example automatically loads and installs the "acro" driver if
*	at least one acro-device is present, and then installs the found
*	number of devices, up to four.
*	After that the found devices are registed in LCC.
*
*	acroN = lcubootAutoDevRegister("/acro0",0x2d,0x1000,1,1,1)
*	acroN = lcubootAutoDevRegister("/acro1",0x2d,0x1400,1,1,1)
*	acroN = lcubootAutoDevRegister("/acro2",0x2d,0x1800,1,1,1)
*	acroN = lcubootAutoDevRegister("/acro3",0x2d,0x1c00,1,1,1)
*	
*	lcubootAutoDrvInstall "acro"
*	
*	lcubootAutoDevCreate "/acro0",0x1000,0xb0,1
*	lcubootAutoDevCreate "/acro1",0x1400,0xb1,1
*	lcubootAutoDevCreate "/acro2",0x1800,0xb2,1
*	lcubootAutoDevCreate "/acro3",0x1c00,0xb3,1
*	
*	lcubootAutoExec acroN,"sysIntEnable",1
*	lcubootAutoSpawn acroN,"tintACRO",200,0x18,2000,"acroInt"
*	...
*	lcubootAutoLccRegisterDevs
*
*
*	A similar - but maybe less elegant - implementation:
*
*	acroN = lcubootAutoProbe(0xffff1000,0xffff1400,0xffff1800,0xffff1c00)
*	lcubootAutoDrvInstall "acro",acroN,"acroDrv","acroDevCreate",acroN,5,80
*	# or with defaults simply: lcubootAutoDrvInstall "acro"
*	lcubootAutoDevCreate "/acro0",0x1000,0xb0,1
*	lcubootAutoDevCreate "/acro1",0x1400,0xb1,1
*	lcubootAutoDevCreate "/acro2",0x1800,0xb2,1
*	lcubootAutoDevCreate "/acro3",0x1c00,0xb3,1
*
*	e=calloc(512,1)
*	lcubootAutoExec acroN>0,"lccRegisterDevice","/acro0",0xffff1000,1,1,1,e
*	lcubootAutoExec acroN>1,"lccRegisterDevice","/acro1",0xffff1400,1,1,1,e
*	lcubootAutoExec acroN>2,"lccRegisterDevice","/acro2",0xffff1800,1,1,1,e
*	lcubootAutoExec acroN>3,"lccRegisterDevice","/acro3",0xffff1c00,1,1,1,e
*	free e
*
*
*   SEE ALSO
*	lcubootAutoEnv(1), lcubootAutoGen(1), lcubootAutoLcc(1), 
*	loadLib(1), symLib(1),
*	ld(2)
*
*   BUGS   
* 
*------------------------------------------------------------------------
*/

#include "lcubootPrivate.h"
#include "sysLib.h"
#include "vme.h"


/*
 * Default suffixes for module-name to yield function names:
 * (they are used when no explicit parameters are given)
 */
const char lcubootAutoDrvFuncSuffix[16] = DRV_FUNC_SUFFIX; /* driver install */
const char lcubootAutoDevFuncSuffix[16] = DEV_FUNC_SUFFIX; /* device install */

/*
 * Default parameters for driver install functions:
 * (they are used when no explicit parameters are given)
 */
int lcubootAutoDrvArg2 = DRV_ARG2;	/* usually number of device channels */
int lcubootAutoDrvArg3 = DRV_ARG3;	/* usually semaphore timeout in ticks*/

/*
 * Device List:
 * (contains data saved by `lcubootAutoDevRegister')
 */
lcubootDEV_LIST lcubootAutoDevList[DEV_LIST_SIZE];
int lcubootAutoDevListCount = 0;

/*
 * Static data:
 */
LOCAL FUNCPTR funcDevCreate = NULL;	/* for lcubootAutoDevCreate */
LOCAL int nextDeviceNumber = 0;		/* for lcubootAutoDevCreate */
LOCAL int maxDeviceCount = 0;		/* for lcubootAutoDevCreate */
LOCAL int devCount = 0; /* number of found devices in lcubootAutoDevRegister */


LOCAL void *getDevProbeAddr(const char *devName)
{
    int i;

    for (i=0; i<lcubootAutoDevListCount; i++)
	{
	if (strcmp(devName, lcubootAutoDevList[i].devName) == 0)
	    return lcubootAutoDevList[i].probeAddr;
	}
    return NULL;
}


int lcubootAutoDevRegister(const char *devName, 
			   int addrSpace, void *busAddr,
			   int lccArg1, int lccArg2, int lccArg3)
{
    char *probeAddr;
    lcubootDEV_LIST *p;

    lcubootLogMsg("Autodev register device %s\n", (int)devName,0,0,0,0,0); 
    /*
     * Check if device-list is already full:
     */
    if (lcubootAutoDevListCount >= (int)NELEMENTS(lcubootAutoDevList))
	{
	errno = lcubootAutoDevListCount;
	RETURN_ABORT("device list overflow", devName);
	}

    /*
     * Convert bus-address to local address:
     */
    if (sysBusToLocalAdrs(addrSpace, busAddr, &probeAddr) == ERROR)
	{
	const char *addrSpaceStr;
	if (VME_AM_IS_SHORT(addrSpace))
	    addrSpaceStr = "A16/SHORT";
	else if (VME_AM_IS_STD(addrSpace))
	    addrSpaceStr = "A24/STD";
	else if (VME_AM_IS_EXT(addrSpace))
	    addrSpaceStr = "A32/EXT";
	else
	    addrSpaceStr = "addrSpace INVALID";
	errno = (int)busAddr; /* just for display */
	RETURN_ABORT("could not convert bus-address", addrSpaceStr);
	}

    /*
     * Probe address, return if nothing there:
     */
    if (lcubootProbeAddress(probeAddr) != lcubootOK)
	return devCount;

    /*
     * Add device to list:
     */
    p = &lcubootAutoDevList[lcubootAutoDevListCount++];
    ++devCount;

    p->devName = devName;
    p->probeAddr = probeAddr;
    p->lccArg1 = lccArg1;
    p->lccArg2 = lccArg2;
    p->lccArg3 = lccArg3;

    return devCount;
}


/*
 * obsolete now for the user, replaced by lcubootAutoDrvInstall:
 *
 *	lcubootAutoDevInit - makes the setup for the following installation of
 *		devices. Defines `funcName' as the function to be used and
 *		`numDevices' as the number of devices to be installed.
 *		It returns lcubootOK, or lcubootERROR if the function is not 
 *		found in the symbol-table.
 *
 */
int lcubootAutoDevInit(const char *funcName, int numDevices)
{
    FUNCPTR funcPtr;

    /*
     * Search for the DevCreate function in the system symbol-table:
     */
    if (lccFindFunctionEntry((char *)funcName,&funcPtr) == ERROR || funcPtr == NULL)
	RETURN_ABORT("function not found", funcName);

    /*
     * Initialize the global data for `lcubootAutoDevCreate':
     */
    nextDeviceNumber = 0;
    maxDeviceCount = numDevices;
    funcDevCreate = funcPtr;
    return lcubootOK;

}


int lcubootAutoDrvInstall(const char *moduleName, int numDevices, 
			 const char *drvFuncName, const char *devFuncName, 
			 int drvArg1, int drvArg2, int drvArg3, 
			 int drvArg4, int drvArg5, int drvArg6)
{
    char drvFnBuf[24];
    char devFnBuf[24];

    /*
     * Don't do anything if no devices are there:
     */
    if (numDevices == 0) /* i.e. defaulted */
	{
	numDevices = devCount; /* take value from DevRegister() */
	if (numDevices == 0)
	    {
	    numDevices = lcubootProbeCount; /* take value from Probe() */
	    }
	}
    devCount = 0;
    lcubootProbeCount = 0;
    if (numDevices <= 0)
	return lcubootOK;

    /*
     * Check and adjust the defaulted parameters:
     */
    if (strlen(moduleName)>sizeof(drvFnBuf)-sizeof(lcubootAutoDrvFuncSuffix) ||
	strlen(moduleName)>sizeof(devFnBuf)-sizeof(lcubootAutoDevFuncSuffix) )
	{
	errno = strlen(moduleName);
	RETURN_ABORT("moduleName too long", moduleName);
	}
    if (drvFuncName == NULL)
	{
	strcpy(drvFnBuf, moduleName);
	strcat(drvFnBuf, lcubootAutoDrvFuncSuffix);
	}
    else strcpy(drvFnBuf,drvFuncName);
    if (devFuncName == NULL)
	{
	strcpy(devFnBuf, moduleName);
	strcat(devFnBuf, lcubootAutoDevFuncSuffix);
	}
    else strcpy(devFnBuf,devFuncName);
    if (drvArg1 == 0)
	drvArg1 = numDevices;
    if (drvArg2 == 0)
	drvArg2 = lcubootAutoDrvArg2;
    if (drvArg3 == 0)
	drvArg3 = lcubootAutoDrvArg3;

    /*
     * Load the driver module:
     */
    errno = lcubootAutoLoad(TRUE, moduleName, 0, 0);
    if (errno != lcubootOK)
	RETURN_ABORT("load failed for driver", moduleName);

    /*
     * Call the Drv function to install the driver:
     */
    errno = lcubootAutoExec(TRUE, drvFnBuf, drvArg1, drvArg2, drvArg3,
			   drvArg4, drvArg5, drvArg6, 0, 0);
    if (errno != lcubootOK)
	RETURN_ABORT("driver installation failed", drvFnBuf);

    /*
     * Setup the data for the device installation:
     */
    if (lcubootAutoDevInit(devFnBuf, numDevices) != lcubootOK)
	RETURN_ABORT("device installation setup failed", devFnBuf);

    return lcubootOK;
}


int lcubootAutoDevCreate(const char *devName, 
			int arg1, int arg2, int arg3, int arg4, int arg5,
			int arg6, int arg7, int arg8, int arg9)
{
    void *probeAddr;

    /*
     * Don't do anything if the device-count has already been exceeded:
     */
    if (nextDeviceNumber >= maxDeviceCount)
	return lcubootOK;

    /*
     * Don't do anything if the device is not there:
     */
    probeAddr = getDevProbeAddr(devName);
    if (probeAddr == NULL || lcubootProbeAddress(probeAddr) != lcubootOK)
	return lcubootOK;

    /*
     * Abort if no install function is defined:
     */
    if (funcDevCreate == NULL)
	RETURN_ABORT("no function to install device", devName);

    /*
     * Create the next device:
     */
    errno = (*funcDevCreate)(devName,
			     arg1,arg2,arg3,arg4,arg5,arg6,arg7,arg8,arg9);
    if (errno != lcubootOK)
	RETURN_ABORT("device installation failed", devName);

    ++nextDeviceNumber;
    return lcubootOK;
}


int lcubootAutoDevCheck(const char *devBaseName, int nominalCount)
{
    char devName[20];
    char *numPtr, *pNameTail;
    DEV_HDR *devPtr;
    int actualCount;

    /*
     * Don't check if nominalCount is below zero:
     */
    if (nominalCount <= 0) return lcubootOK;

    lcubootLogMsg("lcubootAutoDevCheck: %s for %d device%s\n",
		  (int)devBaseName,nominalCount,
		  nominalCount>1?(int)"s":(int)"",0,0,0);

    /*
     * Check parameters:
     */
    if (strlen(devBaseName) >= sizeof(devName)-3)
	{
	errno = strlen(devBaseName);
	RETURN_ABORT("device base name too long",devBaseName);
	}
    if (nominalCount > 99)
	{
	errno = nominalCount;
	RETURN_ABORT("device nominal count too high",devBaseName);
	}

    strcpy(devName, devBaseName);
    numPtr = devName + strlen(devName);

    /*
     * Find matching devices and count them:
     */
    for (actualCount=0; actualCount < nominalCount; actualCount++)
	{
	sprintf(numPtr, "%d", actualCount);
	devPtr = iosDevFind(devName, &pNameTail);
	if (devPtr == NULL || *pNameTail != EOS) break;
	}

    /*
     * Compare the counts:
     */
    if (actualCount != nominalCount)
	{
	errno = actualCount;
	RETURN_ABORT("unexpected device count", devBaseName);
	}
	
    return lcubootOK;
}


/*___oOo___*/
