/******************************************************************************
* E.S.O. - VLT project
*
* "@(#) $Id: lccVxwArch.c,v 1.2 2008/10/27 14:53:57 bjeram Exp $"
*
* who        when       what
* ---------  ---------- ---------------------------------------------------
* P.Duhoux   20/07/98   New
*/

/************************************************************************
*   NAME
*	lccVxwArch - VxWorks independent OS function layer
*
*   SYNOPSIS
*
*   DESCRIPTION
*	These functions hide by overloading the implementation differences
*       on the various LCC platforms (MC68040 & PPC604).
*
*   VARIABLES
*
*   FILES
*
*   ENVIRONMENT
*	CPU_FAMILY - family of the processor (MC680X0 or PPC)
*                    internally set by Makefile from the variable CPU
*
*   COMMANDS
*
*   RETURN VALUES
*
*   CAUTIONS 
*
*   EXAMPLES
*
*   SEE ALSO
*
*   BUGS   
* 
*------------------------------------------------------------------------
*/


static const char *rcsId="@(#) $Id: lccVxwArch.c,v 1.2 2008/10/27 14:53:57 bjeram Exp $"; 

#include "vxWorks.h"
#include "stdio.h"
#include "string.h"
#include "symLib.h"
#include "sysSymTbl.h"
#include "math.h"

/*
 * PUBLIC FUNCTIONS:
 */

/************************************************************************
*   NAME
*	lccSymFindByName, 
*       lccFindFunctionEntry,
*       lccFindSymbolAddress
*       - find symbol address in Global Symbol Table
* 
*   SYNOPSIS
*       #include "lccVxwArch.h"
*
*	STATUS lccSymFindByName(
*           IN  SYMTAB_ID   symTblId,
*           IN  const char  *name,
*           OUT char       **pAddr,
*           OUT SYM_TYPE    *pType)
*
*	STATUS lccFindSymbolAddress(
*           IN  const char  *symName,
*           OUT char       **symAddr)
*
*	STATUS lccFindFunctionEntry(
*           IN  const char  *fctName,
*           OUT FUNCPTR     *fctAddr)
* 
*   DESCRIPTION
*	Idem symFindByName(2) with auto CPU selector.
*       This routine searches in the symbol table <symTblId> for a 
*       symbol matching a specified <name>.
*       If the symbol is found, its value and type are copied to
*       <pValue> and <pType>. 
*
*         <symTblId> IN  : symbol table ID
*         <name>     IN  : symbol name to resolve
*         <pAddr>    OUT : symbol address
*         <pType>    OUT : symbol type
*
*       If the symbol table ID <symTblId> is NULL, the system symbol 
*       table <sysSymTbl> is searched.
*       If <pAddr> or <pType> is NULL, the corresponding value
*       is not returned.
*       If multiple symbols have the same name but differ in type, 
*       the routine chooses the matching symbol most recently
*       added to the symbol table.
*
*       The functions lccFindFunctionEntry() and lccFindSymbolAddress() are 
*       identical to lccSymFindByName() but offer a simpler interface.
*         <name> IN  : function / symbol name to resolve
*         <addr> OUT : function entry / symbol address
*
*   RETURN VALUES
*	OK or ERROR if the symbol table ID is invalid or the symbol cannot be
*       found.
*
*   SEE ALSO
*       symlib, API Programmer's Guide: Object Module Loader
*
*------------------------------------------------------------------------
*/
#define SYMBOL_PREFIX_68K '_'
#define SYMBOL_PREFIX_PPC  0

STATUS lccSymFindByName
          (
          SYMTAB_ID   symTblId,  /* ID of symbol table to look in */
          const char  *name,     /* symbol name to look for       */
          char       **pAddr,    /* where to put symbol value     */
          SYM_TYPE    *pType     /* where to put symbol type      */
          )
{
    const char        *symName = name;
    char              *symAddr;
    SYM_TYPE           symType;

#if (CPU_FAMILY == MC680X0)
    char               tmpName[256];
#endif

    /*
     * Check validity of symbol-table argument
     * - default is VxWorks Global Symbol Table
     */
    if (symTblId == NULL) symTblId = sysSymTbl;

    /*
     * Only for the VxWorks Global Symbol Table
     */
    if (symTblId == sysSymTbl)
	{
	/*
	 * Build symbol name depending on CPU_FAMILY
	 * - prefix _ for MC680X0 only
	 */
#if (CPU_FAMILY == MC680X0)
	if (*name != SYMBOL_PREFIX_68K)
	    {
	    memset(tmpName,0,sizeof(tmpName));
	    *tmpName = SYMBOL_PREFIX_68K;
	    strncat(tmpName,name,sizeof(tmpName)-strlen(tmpName)-1);
	    tmpName[sizeof(tmpName)-1] = 0;
	    symName = tmpName;
	    }
#endif
#if (CPU_FAMILY == PPC)
	if (*name == SYMBOL_PREFIX_68K)
	    symName = name + 1;
#endif
	}

    /*
     * Search for the symbol in the given symbol-table:
     */
    if (symFindByName(symTblId,symName,&symAddr,&symType) == ERROR)
	{
        /* 
	 * printf("Error in symFindByName\n symName: %s\n",symName);
	 */
	return ERROR;
	}
    if (symAddr == NULL)
        {
	/* 
	 * printf("Error symAddr NULL \n symName: %s\n",symName);  
	 */
	return ERROR;
        }

    if (pAddr) *pAddr = (void *)symAddr;
    if (pType) *pType = symType;
    return OK;
}

/************************************************************************
*   NAME
*	lccRound, lccIRound - rounding functions
* 
*   SYNOPSIS
*       #include "lccVxwArch.h"
*
*	double lccRound ( double v )
*	int    lccIRound( double v )
* 
*   DESCRIPTION
*	The routine lccRound rounds a double-precision value <x> to the 
*       nearest integral value.
*       The routine lccIRound rounds a double-precision value <x> to the 
*       nearest integer value.
*
*   RETURN VALUES
*	The routine lccRound returns the double-precision representation 
*       of <x> rounded to the nearest integral value.
*	The routine lccIRound returns The integer nearest to <x>.
*
*   NOTE
*       If <x> is spaced evenly between two integers, lccIRound returns 
*       the even integer.
*
*   SEE ALSO
*       round(3), iround(3)
*       ceil(3), floor(3)
*       math.h, mathALib
*
*------------------------------------------------------------------------
*/
double lccRound ( double x )
{
    if (x < 0.0) 
	return ceil(x - 0.5);
    else
	return floor(x + 0.5);
}

int lccIRound ( double x )
{
    int    i;
    double d;
    if (x < 0.0) 
	{
	i = (int)ceil(x - 0.5);
	d = x - (double)i;
	if ((d == 0.5) && (i % 2)) i++;
	}
    else
	{
	i = (int)floor(x + 0.5);
	d = (double)i - x;
	if ((d == 0.5) && (i % 2)) i--;
	}
    return i;
}

/*___oOo___*/
