/********************************************************************************/
/********************************************************************************/
/*                                                                              */
/* sysVltLib.c for Motorola CPU MV2604 & MV2700                                 */
/*                                                                              */
/********************************************************************************/
/********************************************************************************/
#include "universe.h"
#include "sysVltLib.h"

#include "../all/sysVltLib.c"

/********************************************************************************/
/* WATCHDOG                                                                     */
/********************************************************************************/

/********************************************************************************
* sysVltStartWatchdog - starts the NVRAM (SGS M48T59) watchdog
*
* This routine starts the watchdog. A watchdog timeout will generate
* a system reset.
*
* RETURNS:
*/
#define wdHW_WDTIMER_ADDR   0x1FF7  /* WD Timer control address */

static UINT32 wdTimeout = 0;

STATUS sysVltStartWatchdog ( unsigned int timeout )  /* Watchdog timeout in seconds */
{
    wdTimeout = 0x80 + (timeout << 2) + 2;
    sysNvWrite(wdHW_WDTIMER_ADDR, wdTimeout);
    return OK;
}

/********************************************************************************
* sysVltUpdateWatchdog - update the RTC watchdog
*
* This routine updates the RTC watchdog. If the watchdog is not updated before a 
* timeout it will generate a system reset.
*
* RETURNS:
*/
STATUS sysVltUpdateWatchdog ( void )
{
    sysNvRead(wdHW_WDTIMER_ADDR);              /* Reset timer */
    sysNvWrite(wdHW_WDTIMER_ADDR, wdTimeout);  /* Restart timer */
    return OK;
}

/********************************************************************************/
/* SYSRESET                                                                     */
/********************************************************************************/

/********************************************************************************
* assertSysresetOnVmeBus - reset CPU through SYSRESET on VMEbus
*/

#define CPU_HW_CTRL_REG_ADDR   0xFD050404               /* UNIVERSE_MISC_CTL */
#define CPU_HW_VME_RESET_BIT   0x00004000               /* Bit 14 */
#define CPU_HW_VME_RESET_BITS (CPU_HW_VME_RESET_BIT | \
                               MISC_CTL_SW_LRST     | \
                               MISC_CTL_SW_SRST)        /* Bit 14, 22 & 23 */

void assertSysresetOnVmeBus ( int start )
{
    volatile UINT32 universeMiscCtlReg; 

    UNIV_IN_LONG(UNIVERSE_MISC_CTL,&universeMiscCtlReg);
    universeMiscCtlReg |= CPU_HW_VME_RESET_BITS;
    UNIV_OUT_LONG(UNIVERSE_MISC_CTL,universeMiscCtlReg);
}

/********************************************************************************/
/* BOOT ROM Revision                                                            */
/********************************************************************************/

/********************************************************************************
* romBootRevision - return Boot ROM Revision
*/
static char rbRevision[BOOTROM_REVISION_LEN];
const char *romBootRevision ( void )
{
    sysNvRamGet(rbRevision,BOOTROM_REVISION_LEN,BOOTROM_REVISION_NVRAM_OFFSET);
    return (const char *)rbRevision;
}

/********************************************************************************
* romBootCreation - return Boot ROM Creation date
*/
static char rbCreation[BOOTROM_CREATION_LEN];
const char *romBootCreation ( void )
{
    sysNvRamGet(rbCreation,BOOTROM_CREATION_LEN,BOOTROM_CREATION_NVRAM_OFFSET);
    return (const char *)rbCreation;
}

/********************************************************************************/
/* VME BUS Timeout                                                              */
/********************************************************************************/

/********************************************************************************
 * The VME Bus timeout can take the values:
 *   MISC_CTL_VBTO_DISABLE  = (0 << 28)
 *   MISC_CTL_VBTO_16USEC   = (1 << 28)
 *   MISC_CTL_VBTO_32USEC   = (2 << 28)
 *   MISC_CTL_VBTO_64USEC   = (3 << 28)
 *   MISC_CTL_VBTO_128USEC  = (4 << 28)
 *   MISC_CTL_VBTO_256USEC  = (5 << 28)
 *   MISC_CTL_VBTO_512USEC  = (6 << 28)
 *   MISC_CTL_VBTO_1024USEC = (7 << 28)
 */

/********************************************************************************
* sysVltSetVmeBusTimeout - Set the VME Bus timeout [us]
*/
void sysVltSetVmeBusTimeout ( int vbto )
{
    UINT32 misc_ctl,ctl;
    switch (vbto)
	{
	case    0: ctl = MISC_CTL_VBTO_DISABLE;  break;
	case    8: 
	case   16: ctl = MISC_CTL_VBTO_16USEC;   break;
	case   32: ctl = MISC_CTL_VBTO_32USEC;   break;
	case   64: ctl = MISC_CTL_VBTO_64USEC;   break;
	case  128: ctl = MISC_CTL_VBTO_128USEC;  break;
	case  256: ctl = MISC_CTL_VBTO_256USEC;  break;
	case  512: ctl = MISC_CTL_VBTO_512USEC;  break;
	default:   ctl = MISC_CTL_VBTO_1024USEC; break;
	}
    UNIV_IN_LONG(UNIVERSE_MISC_CTL,&misc_ctl);
    misc_ctl = (misc_ctl & ~MISC_CTL_VBTO_1024USEC) | ctl;
    UNIV_OUT_LONG(UNIVERSE_MISC_CTL,misc_ctl);
}

/********************************************************************************
* sysVltGetVmeBusTimeout - Get the VME Bus timeout [us]
*/
int sysVltGetVmeBusTimeout ( void )
{
    UINT32 misc_ctl,ctl;
    int vbto;

    UNIV_IN_LONG(UNIVERSE_MISC_CTL,&misc_ctl);
    ctl = misc_ctl & ~MISC_CTL_VBTO_1024USEC;

    switch (ctl)
	{
	case MISC_CTL_VBTO_16USEC   : vbto =   16; break;
	case MISC_CTL_VBTO_32USEC   : vbto =   32; break;
	case MISC_CTL_VBTO_64USEC   : vbto =   64; break;
	case MISC_CTL_VBTO_128USEC  : vbto =  128; break;
	case MISC_CTL_VBTO_256USEC  : vbto =  256; break;
	case MISC_CTL_VBTO_512USEC  : vbto =  512; break;
	case MISC_CTL_VBTO_1024USEC : vbto = 1024; break;
	default: vbto =   0; 
	}
    return vbto;
}
    
/********************************************************************************/
/* DMA                                                                          */
/********************************************************************************/
int sysVltDmaPrintEnable = 0;
#define sysVltDmaPrintf \
    if (sysVltDmaPrintEnable) printf
/********************************************************************************
 * Macros for direct access to memory addresses, in units of bytes, words, 
 * or long words.
 */
extern void  sysPciRead32  ();
extern void  sysPciWrite32 ();

#ifndef UNIV_OUT_LONG
#  define UNIV_OUT_LONG(adr,val) \
     sysPciWrite32((vltUINT32)(adr),(vltUINT32)(val))
#endif
#ifndef UNIV_IN_LONG
#  define UNIV_IN_LONG(adr,pVal) \
     sysPciRead32((vltUINT32)(adr),(vltUINT32 *)(pVal))
#endif

LOCAL BOOL sysVltDmaEnabled = FALSE;
LOCAL BOOL sysVltDmaWithInt = FALSE;
LOCAL BOOL sysVltDmaIntEnabled = FALSE;

/***************************************************************************
 * sysVltDmaEnable - Enable DMA transfers
 */
void sysVltDmaEnable ( void )
{
    sysVltDmaEnabled = TRUE;
}

/***************************************************************************
 * sysVltDmaDisable - Disable DMA transfers
 */
void sysVltDmaDisable ( void )
{
    sysVltDmaEnabled = FALSE;
}

/***************************************************************************
 * sysVltDmaIntEnable - Enable interrupt on DMA transfers
 */
void sysVltDmaIntEnable ( void )
{
    if (!sysVltDmaEnabled)
	sysVltDmaIntEnabled = FALSE;
    else
	{
	if (sysVltDmaWithInt)
	    sysVltDmaIntEnabled = TRUE;
	else
	    sysVltDmaIntEnabled = FALSE;
	}
}

/***************************************************************************
 * sysVltDmaDisable - Disable interrupt on DMA transfers
 */
void sysVltDmaIntDisable ( void )
{
    sysVltDmaIntEnabled = FALSE;
}

/***************************************************************************
 * sysVltDmaInit - Configure those DMAC registers which only need to be set once.
 */
void sysVltDmaInit ( BOOL withInt )
{
    UINT32 reg;

    /* 
     * DMA CTL: Use A24 or A32 usr data addr mod, D32 block xfer, always 
     * increment local bus address, optionally increment VME bus address 
     */
    UNIV_IN_LONG(UNIVERSE_PCI_CSR,&reg);
    sysVltDmaPrintf("sysVltDmaInit : UNIVERSE_PCI_CSR = %08X -> ",reg);
    reg |= PCI_CSR_BM;
    UNIV_OUT_LONG(UNIVERSE_PCI_CSR,reg);

    UNIV_IN_LONG(UNIVERSE_DCTL,&reg);
    sysVltDmaPrintf("sysVltDmaInit : UNIVERSE_DCTL = %08X -> ",reg);
    reg = (reg & DCTL_MASK) | 
	  DCTL_VDW_32 | DCTL_VAS_A32 | DCTL_SUPER_SUP | DCTL_PGM_DATA;
    UNIV_OUT_LONG(UNIVERSE_DCTL,reg);
    sysVltDmaPrintf(" %08X\n",reg);

    /* 
     * DMA DGCS : Use VMEbus request level 1, release on BR, non-fair, 
     * direct mode, and start (enable) DMA transfer. 
     * Note: the single-transfer bus requester
     * is programmed for request level 3, and we release the DMAC requester on
     * other bus requests so that single-transfer bus requests can get on the
     * VME bus quickly. 
     */
    reg = DGCS_CHAIN_DMODE | DGCS_VON_16384 | DGCS_VOFF_16;
    reg |= DGCS_STOP | DGCS_HALT | DGCS_DONE | DGCS_LERR | DGCS_VERR | DGCS_P_ERR;
    if (sysVltDmaIntEnabled)
	{
	reg |= DGCS_INT_STOP | DGCS_INT_HALT |
	       DGCS_INT_DONE | DGCS_INT_LERR | 
               DGCS_INT_VERR | DGCS_INT_P_ERR;
	}
    UNIV_OUT_LONG(UNIVERSE_DGCS,reg);
    sysVltDmaPrintf("sysVltDmaInit : UNIVERSE_DGCS =  %08X\n",reg);

    if (withInt)
	{
	/* 
	 * set DMA controller done interrupt level to 6. 
	 * Already configured to use base vector 0x50 (actual vector 0x56) by 
	 * VxWorks init. 
	 */
#if 0
	UNIV_IN_LONG(UNIVERSE_ILR2,&reg);
	sysVltDmaPrintf("sysVltDmaInit : UNIVERSE_ILR2 = %08X -> ",(int)reg);
	reg = (reg & 0xF8FFFFFF) | ILR2_DMA_LEVEL6 ;
	UNIV_OUT_LONG(UNIVERSE_ILR2,reg);
	sysVltDmaPrintf(" %08X\n",reg);
#endif
	sysVltDmaIntEnable();
	}
    else
	sysVltDmaIntDisable();
    sysVltDmaWithInt = withInt;
}

/***************************************************************************/
/*
 * sysVltDmaGetStatus - Tests for completion of DMA transfer started by b016Dma().
 * This uses the 'doDma' flag to make the return value valid even
 * before the first DMA, i.e. otherwise would show 0 (DMA in progress).
 * Return value of 0 indicates DMA not done (in progress).
 * = bit DGCS_ACT of DMAC status register UNIVERSE_DGCS
 * Return value of 1 indicates DMA completed without errors.
 * = bit DGCS_DONE of DMAC status register UNIVERSE_DGCS
 * Any other return value indicates error code as per bits 
 * DGCS_STOP, DGCS_HALT, DGCS_LERR, DGCS_VERR, DGCS_P_ERR of
 * DMAC status register UNIVERSE_DGCS (e.g. bus error).
 */
int sysVltDmaGetStatus ( void )
{
    UINT32 mask = DGCS_STOP | DGCS_HALT | DGCS_LERR | DGCS_VERR | DGCS_P_ERR;
    UINT32 stat;

    if (!sysVltDmaEnabled) 
	{
	sysVltDmaPrintf("sysVltDmaGetStatus : sysVltDmaEnabled=F -> return 1\n");
	return 1;
	}

    UNIV_IN_LONG(UNIVERSE_DGCS,&stat);
    sysVltDmaPrintf("sysVltDmaGetStatus : stat=%X - ",(int)stat);
    if (stat & DGCS_ACT) 
	{
	sysVltDmaPrintf("DGCS_ACT  -> return 0\n");
	return 0;
	}
    if ((stat & (DGCS_DONE | mask)) == DGCS_DONE)
	{
	sysVltDmaPrintf("DGCS_DONE -> return 1\n");
	return 1;
	}
    sysVltDmaPrintf("return %X\n",(int)stat);
    return (int)stat;
}

/***************************************************************************
 * sysVltDmaTransfer - Initiate a DMA transfer between the local bus and the VME bus, in either
 * direction. Returns immediately, use b016DmaDone() to test completion.
 * <src_addr>   IN  is the start address to copy from. It should be a VME 
 *                  address if 'tovme' is FALSE, otherwise it must be a local 
 *                  bus address that does not map to the VME bus. 
 *                  VME addresses in the range 0xF0000000 to 0xF0FFFFFF are 
 *                  mapped to the A24 address space, else uses A32. 
 *                  All addresses must be long-word aligned.
 * <dest_addr>  IN  is the destination address to copy to. It should be a VME
 *                  address if 'tovme' is TRUE, otherwise it must be a local 
 *                  bus address that does not map to the VME bus. 
 *                  VME addresses in the range 0xF0000000 to 0xF0FFFFFF are 
 *                  mapped to the A24 address space, else uses A32. 
 *                  All addresses must be long-word aligned.
 * <count>      IN  is the number of bytes to copy. Must be a multiple of 4.
 * <tovme>      IN  TRUE means copy from local bus to VME bus, 
 *                  else copy from VME busto local bus.
 * <incvme>     IN  TRUE means increment the VME address, else don't.
 * <noblock>    IN  TRUE means don't use block transfer mode (normally does).
 * <isr>        IN  is the interrupt service routine to call when the DMA is 
 *                  complete. This should be the result of a previous 
 *                  intHandlerCreate() call.
 *
 * Return value is error code.
 */

int sysVltDmaTransfer ( UINT32  src_addr, 
		        UINT32  dest_addr, 
		        UINT32  count, 
		        BOOL    tovme, 
		        BOOL    incvme, 
		        BOOL    noblock, 
		        FUNCPTR isr )
{
    int  am;             /* address modifier code to use */
    UINT32 vmeaddr;      /* VME start address            */
    UINT32 pciaddr;      /* PCI start address            */
    UINT32 reg;

    sysVltDmaPrintf("sysVltDmaTransfer(src=%08X dst=%08X cnt=%d tovme=%c incvme=%c noblock=%c isr=%08X\n",
		 (int)src_addr,(int)dest_addr,(int)count,tovme?'T':'F',incvme?'T':'F',noblock?'T':'F',
		 (int)isr);

    /* 
     * check if DMA enabled
     */
    if (!sysVltDmaEnabled) return ERROR;
    if (sysVltDmaIntEnabled && (isr == NULL)) return ERROR;

    /* 
     * check universe DMA operation already in progress 
     */
    if (sysVltDmaGetStatus() == 0) return ERROR;    

    if (tovme)
	{
	vmeaddr = dest_addr;
	pciaddr = src_addr;
	}
    else
	{
	vmeaddr = src_addr;
	pciaddr = dest_addr;
	}

    /* 
     * do some munging on the VME bus address, 
     * 0xF0000000 - 0xF0FFFFFF indicates
     * A24 (STD) address space, else A32 (EXT) 
     */
    if (vmeaddr >= 0xF0000000 && vmeaddr <= 0xF0FFFFFF)  
	{
	am = VME_AM_STD_USR_DATA;
	vmeaddr &= 0x00FFFFFF;
	sysVltDmaPrintf("sysVltDmaTransfer : am=VME_AM_STD_USR_DATA - ");
	}
    else  
	{
	am = VME_AM_EXT_SUP_DATA;
	sysVltDmaPrintf("sysVltDmaTransfer : am = VME_AM_EXT_SUP_DATA - ");
	}
    sysVltDmaPrintf("vme=%08X\n",(int)vmeaddr);

    if (sysVltDmaIntEnabled)
	{
	/* 
	 * set interrupt vector to point at service routine 
	 * (must already be set up as ISR, i.e. 'isr' should be the result of a 
	 * intHandlerCreate() call. 
	 */
#if 0
	sysVltDmaPrintf("sysVltDmaTransfer : intVecSet(%d,isr=%08X)\n",
		     (int)INUM_TO_IVEC(UTIL_INT_VEC_BASE0+LBIV_DMAC),(int)isr);
	intVecSet((FUNCPTR *)INUM_TO_IVEC(UTIL_INT_VEC_BASE0+LBIV_DMAC),isr);
#endif
	}

    /* initialize VME bus address counter */
    sysVltDmaPrintf("sysVltDmaTransfer : UNIVERSE_DVA = %08X\n",(int)vmeaddr);
    UNIV_OUT_LONG(UNIVERSE_DVA,vmeaddr);

    /* initialize local bus address counter */
    sysVltDmaPrintf("sysVltDmaTransfer : UNIVERSE_DLA = %08X\n",(int)pciaddr);
    UNIV_OUT_LONG(UNIVERSE_DLA,pciaddr);
   
    /* initialize VME bus byte counter */
    sysVltDmaPrintf("sysVltDmaTransfer : UNIVERSE_DTBC = %08X\n",(int)count);
    UNIV_OUT_LONG(UNIVERSE_DTBC,count & 0x00FFFFFF);
   
    UNIV_IN_LONG(UNIVERSE_DCTL,&reg);
    sysVltDmaPrintf("sysVltDmaTransfer : UNIVERSE_DCTL = %08X -> ",reg);
    reg = (reg & ~(DCTL_VCT_MSK | DCTL_L2V_MSK)) | 
	  (noblock ? DCTL_VCT_SINGLE  : DCTL_VCT_EN) |    
	  (tovme   ? DCTL_L2V_PCI_VME : DCTL_L2V_VME_PCI);
    UNIV_OUT_LONG(UNIVERSE_DCTL,reg);
    sysVltDmaPrintf(" %08X\n",reg);

    /* 
     * DMA DGCS : Use VMEbus request level 1, release on BR, non-fair, 
     * direct mode, and start (enable) DMA transfer. 
     * Note: the single-transfer bus requester
     * is programmed for request level 3, and we release the DMAC requester on
     * other bus requests so that single-transfer bus requests can get on the
     * VME bus quickly. 
     */
    UNIV_IN_LONG(UNIVERSE_DGCS,&reg);
    reg |= DGCS_STOP | DGCS_HALT | DGCS_DONE | DGCS_LERR | DGCS_VERR | DGCS_P_ERR;
    if (sysVltDmaIntEnabled)
	{
	reg |= DGCS_INT_STOP | DGCS_INT_HALT |
	       DGCS_INT_DONE | DGCS_INT_LERR | 
               DGCS_INT_VERR | DGCS_INT_P_ERR;
	}
    UNIV_OUT_LONG(UNIVERSE_DGCS,reg);
    reg |= DGCS_GO;
    UNIV_OUT_LONG(UNIVERSE_DGCS,reg);
    sysVltDmaPrintf("sysVltDmaTransfer : UNIVERSE_DGCS = %08X\n",reg);

    return OK;
}

/***************************************************************************
 * sysVltDmaClearInt - Clear the DMA interrupt. 
 */
void sysVltDmaClearInt ( void )
{
    UINT32 reg;

    UNIV_IN_LONG(UNIVERSE_DGCS,&reg);
    sysVltDmaPrintf("sysVltDmaClearInt : UNIVERSE_DGCS = %08X\n",reg);
    reg |= DGCS_STOP | DGCS_HALT | DGCS_DONE | DGCS_LERR | DGCS_VERR | DGCS_P_ERR;
    UNIV_OUT_LONG(UNIVERSE_DGCS,reg);
}

/***************************************************************************
 * sysVltDmaExit - Tidy up DMA
 */
void   sysVltDmaExit ( void )
{
    ;
}

/********************************************************************************/
/* PCI Bus Control                                                              */
/********************************************************************************/

/***************************************************************************
 * sysVltPciGetIntLvlA - Get Interrupt Level A
 */
int sysVltPciGetIntLvlA ( int bus, int dev, int fct ) 
{ 
    return PMC_INT_LVL1; 
}

/***************************************************************************
 * sysVltPciGetIntVecA - Get Interrupt Vector A
 */
int sysVltPciGetIntVecA ( int bus, int dev, int fct ) 
{ 
    return PCI_PRI_INTA_VEC; 
}

/***************************************************************************
 * sysVltGetPciBaseAddress - Get PCI base address
 */
UINT32 sysVltGetPciBaseAddress ( int bus, int dev, int fct )
{
    return (UINT32)PCI_MEM_ADRS + PMC_DEV_SPACE;  
}

/***************************************************************************
 * sysVltLocal2PciMemAdrs - Convert SRAM adress to PCI Memory space
 */
UINT32 sysVltLocal2PciMemAdrs ( UINT32 localAdr )
{ 
    return localAdr; 
}

/***************************************************************************
 * sysVltConnectPciInterrupt - Connect PCI Interrupt
 */
STATUS sysVltConnectPciInterrupt (VOIDFUNCPTR *iv, VOIDFUNCPTR isr, int par )
{
    return intConnect(iv,isr,par);
}

/***************************************************************************
 * sysVltEnablePciInterrupt - Enable PCI Interrupt
 */
STATUS sysVltEnablePciInterrupt ( int intLevel )
{
    return intEnable(intLevel);
}

/***************************************************************************
 * sysVltDisablePciInterrupt - Disable PCI Interrupt
 */
STATUS sysVltDisablePciInterrupt ( int intLevel )
{
    return intDisable(intLevel);
}

/***************************************************************************
 * sysVltMemProbe - vxMemProbe 
 */
STATUS sysVltMemProbe
(
char * adrs, /* address to be probed */
int mode,    /* VX_READ or VX_WRITE */
int length,  /* 1, 2, or 4 */
char * pVal  /* where to return value, or ptr to value to be written */
)
{
    return vxMemProbe (adrs, mode, length, pVal);
}

/* ___oOo___ */










