/* fdcDrv.c - Floppy Disk Controller (FDC) Input/Output Driver Module */

/* Copyright 1984-2001 Wind River Systems, Inc. */
/* Copyright 1996,1997,1998 Motorola, Inc., All Rights Reserved */

/*
modification history
--------------------
01c,16sep01,dat  Use of WRS_ASM macro
01b,15apr98,dat  added INCLUDE_FD, using macros FD_BASE_ADDR and
		 FD_DMA_CHAN to match old API for fdDrv().
01a,12dec97,rbb  created by Motorola.
*/

/*
DESCRIPTION
This is the I/O driver for a standard PS2 floppy device controller (FDC).
This driver is normally included as a source file, and expects the macro
INCLUDE_FD to be defined if any code is to be compiled.

USER CALLABLE ROUTINES
Most of the routines in this driver are accessible only through the I/O
system.  Two routines, however, must be called directly: fdDrv() to
initialize the driver, and fdDevCreate() to create devices.
Before using the driver, it must be initialized by calling the fdDrv().
The macros FD_BASE_ADDR and FD_DMA_CHAN are needed to provide the base
address of the floppy disk controller device and the DMA channel number
to be used. (This is to maintain compatibility with the existing API for
fdDrv() ).

.CS
STATUS fdDrv
    (
    UINT intVector,		// interrupt vector
    UINT intLevel		// interrupt level
    )
.CE

This routine initializes the floppy disk controller driver, sets up
interrupt vectors, and performs hardware initialization of the floppy
disk controller chip.

This routine should be called exactly once, before any reads, writes,
or calls to fdDevCreate().  Normally, it is called from usrRoot() in
usrConfig.c.

CREATING DISK DEVICES
Before a disk device can be used, it must be created.  This is done
with the fdDevCreate() call.  Each floppy disk drive to be used may
have one or more devices associated with it, by calling this routine.
The way the device is created with this call determines the floppy
diskette type, and whether it covers the whole disk or just part of
it.

fdDevCreate() - create a device for a floppy disk

.CS
BLK_DEV *fdDevCreate
    (
    UINT driveNumber,		// drive number (0 to 3)
    UINT fdType,		// type of floppy disk (table index)
    UINT nBlocks,		// device size in blocks (0 = whole disk)
    UINT blkOffset 		// offset from start of device
    )
.CE

This routine creates a device for a specified floppy disk.

The driveNumber parameter is the drive number of the floppy disk; valid
values are 0 to 3.

The fdType parameter specifies the type of diskette, which is described
in the structure table fdcTypes[] in fdcDrv.c.  fdcType is an index
into the table.  Currently the table contains two diskette types:

 - An fdcType of 0 indicates the first entry in the table (3.5" 2HD, 1.44MB);
 - An fdcType of 1 indicates the second entry in the table (3.5" 2HD, 720KB);

Members of the fdcTypes[] structure are:

   UINT numberofheads;		// number of heads (sides)
   UINT numberoftracks;		// number of tracks
   UINT sectorstrack;		// sectors (blocks) per track
   UINT sectorsize;		// sector (block) size in bytes
   UINT ratestep;		// step rate
   UINT ratedata;		// data rate
   UINT gaprw;			// read/write sector gap
   UINT gapformat;		// format gap

The nBlocks parameter specifies the size of the device, in blocks.  If
nBlocks is zero, the whole disk is used.

The blkOffset parameter specifies an offset, in blocks, from the start
of the device to be used when writing or reading the floppy disk.  This
offset is added to the block numbers passed by the file system during
disk accesses.  (VxWorks file systems always use block numbers beginning
at zero for the start of a device.)  Normally, blkOffset is 0.


For instance, to create a rawFs compatible device, covering the whole
disk (3.5" 2HD, 1.44MB) on drive 0, the proper call would be:
.CS
    fdDevCreate (0, 0, 0, 0);
.CE

IOCTL
This driver responds to all the same ioctl codes as a normal block
device driver.  To format a disk use FIODISKFORMAT.

CAVEATS
No mechanism exists for detecting the change of media.  The driver
does not support the concurrent operation of multiple drives.  This
is due to the fact that all drives share the same DMA channel.
*/


/*
 * includes
 *
 * include file paths are dependent on makefile
 */

#include "vxWorks.h"	/* vxWorks generics */
#include "taskLib.h"	/* taskLib functions */
#include "ioLib.h"	/* input/output generics */
#include "blkIo.h"	/* block input/output specifics */
#include "semLib.h"	/* semaphore operations */
#include "cacheLib.h"	/* cache control */
#include "intLib.h"	/* interrupt control */
#include "errnoLib.h"	/* error number */
#include "string.h"

#include "fdc.h"	/* floppy disk controller (FDC) */
#include "fdcDrv.h"	/* floppy disk controller (FDC) driver header */
#include "i8237.h"	/* i8237 DMA controller device */

#ifdef INCLUDE_FD	/* do nothing, if not defined */

/* defines */

#ifndef EIEIO_SYNC
# define EIEIO_SYNC  _WRS_ASM(" eieio; sync")
#endif  /* EIEIO_SYNC */

/* typedefs */

/* globals */

LOCAL UINT fdcDrvInitFlag = 0;		/* driver initialization flag */
LOCAL UINT fdcDrvBaseAddress = 0;	/* base address of FDC */
LOCAL UINT fdcDrvIntVector = (UINT)-1;	/* interrupt vector number */
LOCAL UINT fdcDrvIntLevel = (UINT)-1;	/* interrupt level number */
LOCAL UINT fdcDrvDmaChannel = (UINT)-1;	/* DMA channel number */
LOCAL SEM_ID fdcDrvSemId;		/* driver ownership semaphore */
LOCAL FDC_IARGS fdcDrvIArgs;		/* interrupt handler arguments */
LOCAL UINT fdcDrvIntCount[FDC_NDRIVES] = { 0 };

/* locals */

LOCAL FDC_TYPE fdcTypes[] = {
   { 2, 80, 18, 512, 0, 500, 0x1B, 0x6C },	/* 1.44MB */
   { 2, 80,  9, 512, 0, 250, 0x2A, 0x50 }	/* 720KB */
};
#define FDC_TYPES_TABLESIZE	(sizeof(fdcTypes)/sizeof(FDC_TYPE))

/* forward declarations */

extern STATUS fdDrv();
extern BLK_DEV *fdDevCreate();
LOCAL STATUS fdcIoctl();
LOCAL STATUS fdcRead();
LOCAL STATUS fdcWrite();
LOCAL STATUS fdcStatusChk();

LOCAL UINT fdcCheck();		/* check state */
LOCAL UINT fdcClearReset();	/* clear reset state */
LOCAL void fdcDelay();		/* delay (sleep) in milli-seconds */
LOCAL UINT fdcDrvMain();	/* fdc driver main entry point */
LOCAL UINT fdcFormat();		/* format track/disk */
LOCAL UCHAR fdcDRCode();	/* return data-rate code */
LOCAL UINT fdcInit();		/* initialize operating parameters */
LOCAL void fdcInt();		/* interrupt handler */
LOCAL void fdcSetAES();	/* setup additional error status information */
LOCAL UINT fdc_clsn();		/* calculate logical sector number */
LOCAL void fdcCP();		/* calculate position */
LOCAL UINT fdcRxd();		/* receive byte from data FIFO */
LOCAL UINT fdcRxdEp();		/* receive data from FIFO, execution phase */
LOCAL UINT fdcSeek();		/* seek to track */
LOCAL void fdcStat();	/* retrieve command execution status (results) */
LOCAL UINT fdcTxd(FDC *, UCHAR);	/* transmit (send) byte to data FIFO */
LOCAL UINT fdcTxdEp();		/* transmit data to FIFO, execution phase */
LOCAL void fdcExeCmd();		/* execute command */
LOCAL UINT fdcXfer();		/* read/write transfer */
LOCAL UINT fdcXfrcheck();	/* sanity check on block size/number of blocks/sector size parameters */
LOCAL UINT fdcSCode();		/* return sector code */

/* externals */

extern void bzero();		/* zero or clear memory routine */
extern void *calloc();		/* memory allocate and clear */
extern void *memalign();	/* aligned memory allocate */
extern int sysClkRateGet();	/* system clock rate */

extern void isaDmaInit();	/* initialize DMA (global configuration) */
extern void isaDmaStart();	/* initialize DMA channel for transfer */
extern void isaDmaStop();	/* disable DMA channel */
extern UINT isaDmaStatus();	/* query DMA channel for transfer status */

/*******************************************************************************
*
* fdDrv - initialize the floppy disk driver
* 
* This function's purpose is to the initialize the floppy disk
* controller device for operation.
*
* The macros FD_BASE_ADDR and FD_DMA_CHAN must be defined for this driver
* to compile correctly.
*
* RETURNS: OK, or ERROR if driver initialization fails
*/

STATUS fdDrv
    (
    register UINT interruptVector,	/* interrupt vector */
    register UINT interruptLevel	/* interrupt level */
    )
    {
    register UINT baseAddress = FD_BASE_ADDR;	/* base address of FDC */
    register UINT dmaChannel = FD_DMA_CHAN; 	/* DMA channel number */

    /* initialize driver, if initialized, return ERROR */

    if (fdcDrvInitFlag) 
	{
        return (ERROR);
   	}

    /* make some sanity of the desired mode */

    if (interruptVector != (UINT)-1) 
	{
        if (dmaChannel == (UINT)-1) 
	    {
   	    return (ERROR);
      	    }
   	}

    /* create semaphore to interlock access to the FDC */

    if ((fdcDrvSemId = semBCreate(SEM_Q_PRIORITY, SEM_FULL)) == (SEM_ID)NULL) 
	{
        return (ERROR);
   	}

    /* connect/enable interrupt handler */

    if (interruptVector != (UINT)-1) 
	{
        intConnect( (VOIDFUNCPTR *)interruptVector,
		    (VOIDFUNCPTR)fdcInt,
		    (int)((FDC_IARGS *)&fdcDrvIArgs) );
        intEnable(interruptVector);
	}

    /*
     * initialize the base address of FDC variable, user may override
     * if the value is not zero
     */

    if (!fdcDrvBaseAddress) 
	{
        fdcDrvBaseAddress = baseAddress;
	} 

    /*
     * initialize the DMA channel number for this FDC
     * initialize the interrupt vector/level
     */

    fdcDrvDmaChannel = dmaChannel;
    fdcDrvIntVector = interruptVector;
    fdcDrvIntLevel = interruptLevel;

    /* set the driver initialization flag to a true state */

    fdcDrvInitFlag = (UINT)-1;
	
    return (OK);
    }

/*******************************************************************************
*
* fdDevCreate - create a device for a floppy disk
* 
* This function's purpose is to create a device for a specified
* floppy disk.
*
* RETURNS: pointer to BLK_DEV structure
*/

BLK_DEV * fdDevCreate
    (
    register UINT driveNumber,	/* drive number (0 to 3) */
    register UINT fdType,	/* type of floppy disk (table index) */
    register UINT nBlocks,	/* device size in blocks (0 = whole disk) */
    register UINT blkOffset 	/* offset from start of device */
    )
    {
    register FDC_DEV *pDev;		/* device descriptor pointer */
    register UINT dmaBufferAddress;	/* DMA buffer address */
    register UINT realBlocks;		/* real number of blocks */

    /* do not create device if the driver has not been initialized */

    if (!fdcDrvInitFlag) 
	{
        return ((BLK_DEV *)NULL);
	}

    /* validate floppy disk drive number */

    if (driveNumber > (FDC_NDRIVES - 1)) 
	{
        return ((BLK_DEV *)NULL);
        }

    /* validate floppy device/disk type */

    if (fdType >= FDC_TYPES_TABLESIZE) 
	{
        return ((BLK_DEV *)NULL);
        }

    /* allocate memory for device descriptor */

    pDev = calloc(1, sizeof(FDC_DEV));
    if (pDev == (FDC_DEV *)NULL) 
	{
        return ((BLK_DEV *)NULL);
        }
   
    /* allocate DMA buffer */

    dmaBufferAddress = (UINT)memalign(FDC_DMAALIGN, FDC_DMAALIGN);
    if (dmaBufferAddress == (UINT)NULL) 
	{
        return ((BLK_DEV *)NULL);
        }
    pDev->dmaBuffer = dmaBufferAddress;

    /* create interrupt level to task level semaphore */

    if ((pDev->intSemId = semBCreate(SEM_Q_FIFO, SEM_EMPTY)) == (SEM_ID)NULL) 
	{
      	return ((BLK_DEV *)NULL);
   	}

    /* initialize device specific data structures */

    pDev->driveNumber = driveNumber;
    pDev->fdcType = fdcTypes[fdType];

    /*
     * initialize block information, perfrom some sanity
     * checks on the blkOffset and nBlocks arguments
     */

    pDev->blockSize = pDev->fdcType.sectorsize;
    pDev->blockOffset = blkOffset;
    realBlocks = pDev->fdcType.numberofheads *
                 pDev->fdcType.numberoftracks *
 		 pDev->fdcType.sectorstrack;
    if ((nBlocks > realBlocks) ||
        (blkOffset >= realBlocks) ||
        ((blkOffset + nBlocks) > realBlocks)) 
	{
      	return ((BLK_DEV *)NULL);
   	}
    if (!nBlocks) 
	{
        pDev->blockTotal = realBlocks - blkOffset;
   	} 
    else 
	{
        pDev->blockTotal = nBlocks;
   	}

    /* initialize device block descriptor */

    pDev->fdcBlockDev.bd_blkRd = fdcRead;
    pDev->fdcBlockDev.bd_blkWrt = fdcWrite;
    pDev->fdcBlockDev.bd_ioctl = fdcIoctl;
    pDev->fdcBlockDev.bd_reset = NULL;
    pDev->fdcBlockDev.bd_statusChk = fdcStatusChk;
    pDev->fdcBlockDev.bd_removable = TRUE;
    pDev->fdcBlockDev.bd_nBlocks = realBlocks;
    pDev->fdcBlockDev.bd_bytesPerBlk = pDev->fdcType.sectorsize;
    pDev->fdcBlockDev.bd_blksPerTrack = pDev->fdcType.sectorstrack;
    pDev->fdcBlockDev.bd_nHeads = pDev->fdcType.numberofheads;
    pDev->fdcBlockDev.bd_retry = 0;
    pDev->fdcBlockDev.bd_mode = O_RDWR;
    pDev->fdcBlockDev.bd_readyChanged = TRUE;

    /* initialize the system clock rate */

    pDev->sysClkRate = sysClkRateGet();

    return ((BLK_DEV *)pDev);
    }

/*******************************************************************************
*
* fdcRead - fdc read blocks
*
* This function's purpose is to read the specified number of
* blocks from the specified device.
*
* RETURNS: OK, or ERROR if the read failed
*/

LOCAL STATUS fdcRead
    (
    register FDC_DEV *pDev,		/* pointer to device descriptor */
    register UINT startBlk,		/* starting block to read */
    register UINT numBlks,		/* number of blocks to read */
    register char *pBuf 		/* pointer to buffer to receive data */
    )
    {
    register UINT localStatus;	/* local status variable */

    FDC_CMDPCKT fdcCmdPacket;	/* command packet */

    /*
     * verify the request against the logical block parameters (all
     * or nothing)
     */

    if ((startBlk + pDev->blockOffset + numBlks) > pDev->blockTotal) 
	{
        return (ERROR);
	}

    /* check for a NOP request */

    if (!numBlks) 
	{
        errno = S_ioLib_DEVICE_ERROR;
        return (ERROR);
	}

    /* build command packet */
 
    fdcCmdPacket.command = FDC_READOP;
    fdcCmdPacket.status = 0;
    fdcCmdPacket.memaddr = (UINT)pBuf;
    fdcCmdPacket.blcknum = startBlk + pDev->blockOffset;
    fdcCmdPacket.nblcks = numBlks;
    fdcCmdPacket.tdflg = 0;
    fdcCmdPacket.aescount = 0;
 
    /* take ownership, call driver, release ownership */

    semTake(fdcDrvSemId, WAIT_FOREVER);
    localStatus = fdcDrvMain(pDev, (FDC_CMDPCKT *)&fdcCmdPacket);
    semGive(fdcDrvSemId);

    return ((localStatus ? ERROR : OK));
    }

/*******************************************************************************
*
* fdcWrite - fdc write blocks
* 
* This function's purpose is to write the specified number of
* blocks to the specified device.
*
* RETURNS: OK, or ERROR if the write failed
*/

LOCAL STATUS fdcWrite
    (
    register FDC_DEV *pDev,		/* pointer to device descriptor */
    register UINT startBlk,		/* starting block to write */
    register UINT numBlks,		/* number of blocks to write */
    register char *pBuf 		/* pointer to buffer of send data */
    )
    {
    register UINT localStatus;	/* local status variable */

    FDC_CMDPCKT fdcCmdPacket;	/* command packet */

    /*
     * verify the request against the logical block parameters (all
     * or nothing)
     */

    if ((startBlk + pDev->blockOffset + numBlks) > pDev->blockTotal) 
	{
        return (ERROR);
	}

    /* check for a NOP request */

    if (!numBlks) 
	{
        errno = S_ioLib_DEVICE_ERROR;
        return (ERROR);
	}

    /* build command packet */

    fdcCmdPacket.command = FDC_WRITOP;
    fdcCmdPacket.status = 0;
    fdcCmdPacket.memaddr = (UINT)pBuf;
    fdcCmdPacket.blcknum = startBlk + pDev->blockOffset;
    fdcCmdPacket.nblcks = numBlks;
    fdcCmdPacket.tdflg = 0;
    fdcCmdPacket.aescount = 0;

    /* take ownership, call driver, release ownership */

    semTake(fdcDrvSemId, WAIT_FOREVER);
    localStatus = fdcDrvMain(pDev, (FDC_CMDPCKT *)&fdcCmdPacket);
    semGive(fdcDrvSemId);

    return ((localStatus ? ERROR : OK));
    }

/*******************************************************************************
*
* fdcIoctl - fdc i/o control
*
* This function's purpose is to perform the specified I/O control
* operation.
*
* RETURNS: OK, or ERROR if ioctl request failed
*/

LOCAL STATUS fdcIoctl
    (
    register FDC_DEV *pDev,	/* pointer to device descriptor */
    register int funcCode,	/* ioctl() function code */
    register int arg 		/* function-specific argument */
    )
    {
    register UINT localStatus;	/* local status variable */

    FDC_CMDPCKT fdcCmdPacket;	/* command packet */

    switch (funcCode) 
	{
        case FIODISKFORMAT:

	    /* build command packet */

	    fdcCmdPacket.command = FDC_FRMTOP;
	    fdcCmdPacket.status = 0;
	    fdcCmdPacket.memaddr = 0;
	    fdcCmdPacket.blcknum = 0;
	    fdcCmdPacket.nblcks = 0;
	    fdcCmdPacket.tdflg = 'D';
	    fdcCmdPacket.aescount = 0;

	    /* take ownership, call driver, release ownership */

	    semTake(fdcDrvSemId, WAIT_FOREVER);
	    localStatus = fdcDrvMain(pDev, (FDC_CMDPCKT *)&fdcCmdPacket);
	    semGive(fdcDrvSemId);

	    break;
        default:
	    errno = S_ioLib_UNKNOWN_REQUEST;
	    localStatus = ERROR;
	    break;
	}

    return ((localStatus ? ERROR : OK));
    }

/*******************************************************************************
*
* fdcStatusChk - fdc status check
*
* This function's purpose is to perform a status check on the
* specified device.  The status check basically checks to see
* if the disk has been changed.
*
* RETURNS: OK, or ERROR if check fails
*/

LOCAL STATUS fdcStatusChk
    (
    register FDC_DEV *pDev		/* pointer to device descriptor */
    )
    {
    register STATUS lstatus;	/* local error status */
    register FDC *pFdc;		/* FDC registers pointer */

    FDC_CMDPCKT fdcCmdPacket;	/* command packet */
    FDC_CRDATA fdc_crdata;

    /*
     * initialize pointer to FDC registers
     * initialize local error status variable
     */

    pFdc = (FDC *)fdcDrvBaseAddress;
    lstatus = OK;

    /* take ownership */

    semTake(fdcDrvSemId, WAIT_FOREVER);

    /* motor on */

    pFdc->dor |= (pDev->driveNumber | (0x10 << pDev->driveNumber));
    EIEIO_SYNC;

    /*
     * query the digital input register for a disk change status, if
     * a disk change has occurred, set the "ready change" flag
     */

    if (pFdc->dir_ccr & FDC_DIR_PS2_DSKCHG) 
	{
   
        /* set the "ready change" flag */

        pDev->fdcBlockDev.bd_readyChanged = TRUE;

        /* build command packet */

        fdcCmdPacket.command = FDC_CHCKOP;
        fdcCmdPacket.status = 0;
        fdcCmdPacket.memaddr = 0;
        fdcCmdPacket.blcknum = 0;
        fdcCmdPacket.nblcks = 0;
        fdcCmdPacket.tdflg = 0;
        fdcCmdPacket.aescount = 0;

        /* initialize interrupt handler arguments data structure */

        fdcDrvIArgs.pDev = pDev;
        fdcDrvIArgs.pCmd = (FDC_CMDPCKT *)&fdcCmdPacket;
        fdcDrvIArgs.pFdc = pFdc;
        fdcDrvIArgs.cr_data_p = (FDC_CRDATA *)&fdc_crdata;

        /* check state (take to known state) */

        if (fdcCheck(pDev, (FDC_DEV *)&fdcCmdPacket, pFdc, fdc_crdata, 1)) 
	    {
	    lstatus = ERROR;
            }
   	}

    /* motor off */

    pFdc->dor &= (FDC_DOR_DMAEN|FDC_DOR_RESET);

    /* release ownership */

    semGive(fdcDrvSemId);

    return (lstatus);
    }

/*******************************************************************************
*
* fdcDrvMain - fdc driver main entry point
*
* This function's purpose is the main entry point into the FDC
* Floppy Disk I/O driver.
* 
* RETURNS: 0 if OK, non-zero for error condition which means the status
*  word in the command packet is set.
*/

LOCAL UINT fdcDrvMain
    (
    register FDC_DEV *pDev,	/* device descriptor pointer */
    register FDC_CMDPCKT *pCmd 	/* command packet pointer */
    )
    {
    register UINT lstatus;	/* local error status */

    FDC_CRDATA fdc_crdata;

    /* setup local error status (error condition) */

    lstatus = (UINT)-1;

    /* initialize interrupt handler arguments data structure */

    fdcDrvIArgs.pDev = pDev;
    fdcDrvIArgs.pCmd = pCmd;
    fdcDrvIArgs.pFdc = (FDC *)fdcDrvBaseAddress;
    fdcDrvIArgs.cr_data_p = (FDC_CRDATA *)&fdc_crdata;

    /* perform the requested action */

    switch (pCmd->command) 
	{
        case FDC_READOP:
        case FDC_WRITOP:
	    /*
	     * sanity check of block size/number of blocks/physical
	     * sector size
	     */

	    if ((pCmd->command == FDC_READOP) || (pCmd->command == FDC_WRITOP)) 
		{
	        if (fdcXfrcheck(pDev, pCmd, 0)) 
		    {
	            break;
	    	    }
	 	}
	    if (!(fdcXfer(pDev, pCmd, (FDC *)fdcDrvBaseAddress, fdc_crdata))) 
		{
	        lstatus = 0;
	 	}
	    break;

	case FDC_FRMTOP:
	    if (!(fdcFormat(pDev, pCmd, (FDC *)fdcDrvBaseAddress, fdc_crdata)))
		{
	    	lstatus = 0;
	 	}
	    break;
	}
    if (lstatus) 
	errno = S_ioLib_DEVICE_ERROR;

    return (lstatus);
    }

/*******************************************************************************
*
* fdcXfer - read/write transfer
*
* This function's purpose is to perform the specified data
* transfers with the specified device.  The data requests
* are blocked to a track size, only a one track can be
* read/written at a time.
*
* RETURNS: 0 if OK, non-zero for error condition which means the status
*  word in the command packet is set.
*/

LOCAL UINT fdcXfer
    (
    register FDC_DEV *pDev,		/* device descriptor pointer */
    register FDC_CMDPCKT *pCmd,		/* command packet pointer */
    register FDC *pFdc,			/* FDC registers pointer */
    register FDC_CRDATA *cr_data_p 	/* command/result data pointer */
    )
    {
    register UINT bufadd;		/* starting address of data transfer */
    register UINT t_count, m_count;	/* data count variables */
    register UINT d_count, x_count;	/* data count variables */
    register UINT l_count;		/* loop count variable */
    register UINT lstatus;		/* local status flag */
    register UCHAR statusx;		/* status register copy */
    register UINT e_sector;		/* ending sector numbner */
    register UCHAR *pd_p;		/* phase data pointer */

    struct fdc_pos pos;

    /* setup the return status */

    lstatus = 0;

    /*
     * calculate starting logical sector number of data
     * transfer, this is only done once
     */

    pos.lsector =
        fdc_clsn( pCmd->blcknum, pDev->fdcType.sectorsize,
	          pDev->blockSize, pDev->fdcType.sectorstrack, 1 );

    /*
     * calculate the number of bytes in a track, this is the
     * maximum number of bytes that can be transferred at one
     * given time
     *
     * calculate the number of bytes that the user specified
     */

    t_count = pDev->fdcType.sectorsize * pDev->fdcType.sectorstrack;
    m_count = pCmd->nblcks * pDev->blockSize;

    /* check the sanity of number of blocks requested */

    if (((pos.lsector * pDev->fdcType.sectorsize) + m_count) >
        (t_count * pDev->fdcType.numberofheads * pDev->fdcType.numberoftracks)) 
	{
        pCmd->status = FDC_ERROR_ILLDREQ;
        return ((UINT)-1);
   	}

    /* check state (take to known state) */

    if (fdcCheck(pDev, pCmd, pFdc, cr_data_p, 0)) 
	{
        lstatus = (UINT)-1;
        m_count = 0;
	}

    /*
     * setup starting address of data transfer
     * read/write data tracks loop
     */

    bufadd = pCmd->memaddr;
    for (l_count = 0; m_count; l_count++) 
	{
   
      	/* calculate floppy disk address from logical sector number */

      	fdcCP( pDev->fdcType.numberofheads,
	     pDev->fdcType.sectorstrack,
	     (struct fdc_pos *)&pos );

      	/*
         * seek to track for initial data transfer
         *
         * if the seek to track call fails, post error status and
         * terminate for loop
         */

      	if (l_count == 0) 
	    {
	    if (fdcSeek(pDev, pCmd, pFdc, cr_data_p, pos.headno, pos.cylndrno)) 
		{
	        lstatus = (UINT)-1;
	        break;
	 	}
      	    }

      	/*
         * calculate the size (number of bytes) for this transfer
         * calculate the ending sector number for this transfer
         */

      	if (pos.sectorno > 1) 
	    {
	    d_count = 
		((pDev->fdcType.sectorstrack - 
		 (pos.sectorno - 1)) * pDev->fdcType.sectorsize);
	    if (d_count >= m_count) 
		{
	    	d_count = m_count;
	    	e_sector = 
		    ((d_count / pDev->fdcType.sectorsize) - 1) + pos.sectorno;
	 	} 
	    else 
		{
	        e_sector = pDev->fdcType.sectorstrack;
	 	}
      	    } 
	else 
	    {
	    if (m_count >= t_count) 
		{
	        d_count = t_count;
	 	} 
	    else 
		{
	        d_count = m_count;
		}
	   e_sector = d_count / pDev->fdcType.sectorsize;
      	   }

      	/* setup the "read-data" or "write-data" command structure */

      	pd_p = (UCHAR *)&cr_data_p->c_data.c_justdata.databytes[0];
      	bzero((UCHAR *)cr_data_p, sizeof(FDC_CRDATA));
      	pd_p[0] = FDC_CP_MFM | 
		  ((pCmd->command == FDC_READOP) ? 
		   FDC_CS_READDATA : FDC_CS_WRITEDATA);
      
        pd_p[1] = pDev->driveNumber | (pos.headno ? FDC_ST0_HDS : 0);
        pd_p[2] = pos.cylndrno;
        pd_p[3] = pos.headno;
        pd_p[4] = pos.sectorno;
        pd_p[5] = fdcSCode(pDev->fdcType.sectorsize);
        pd_p[6] = (UCHAR)e_sector;
        pd_p[7] = (UCHAR)pDev->fdcType.gaprw;
        pd_p[8] = 0xFF;

        /* setup/start DMA controller */

        if (fdcDrvDmaChannel != (UINT)-1) 
	    {

	    /* copy caller's buffer to internal buffer */

	    if (pCmd->command == FDC_WRITOP) 
		{
	        memcpy((void *) pDev->dmaBuffer, (void *) bufadd, d_count);
	        cacheFlush(DATA_CACHE, (void *)pDev->dmaBuffer, d_count);
	 	}

	    /* initialize DMA controller for DMA data transfer */

	    isaDmaStart( fdcDrvDmaChannel,
		         I8237_MODE_TM_DEMAND,
		         ((pCmd->command == FDC_READOP) ? 
			  I8237_MODE_TT_WRITE : I8237_MODE_TT_READ),
		         pDev->dmaBuffer, d_count );
	    }

	/* issue the "read-data" or the "write-data" command */

      	fdcExeCmd(pFdc, pd_p, sizeof(cr_data_p->c_data.c_readdata));

      	/* wait for interrupt, or poll for completion */

      	if (fdcDrvIntVector != (UINT)-1) 
	    {
	    if (semTake(pDev->intSemId, pDev->sysClkRate * 2) == ERROR) 
		{
	        pFdc->fifo = 0x00;
	        EIEIO_SYNC;
	        isaDmaStop(fdcDrvDmaChannel);
	        x_count = 0;
	 	} 
	    else 
		{
	        x_count = d_count;
	 	}
      	    } 
	else 
	    {

	    /* poll DMA controller for completion (terminal count) */

	    if (fdcDrvDmaChannel != (UINT)-1) 
		{
	        for (x_count = 0;; x_count++) 
		    {

			/* sleep for a bit (10ms) */

	       		fdcDelay(10);

			/* retrieve DMA controller status */

	       		if ((statusx = 
				(UCHAR)isaDmaStatus(fdcDrvDmaChannel)) == 1) 
			    {
		  	    x_count = d_count;
		  	    break;
	       		    } 
			else 
			    {
		  	    if (x_count >= (2000/10)) 
				{
		     		pFdc->fifo = 0x00;
		     		EIEIO_SYNC;
		     		isaDmaStop(fdcDrvDmaChannel);
		     		x_count = 0;
		    		break;
		  		}
	       		    }
	    		}
	 	    } 
		else 
		    {

	    	    /*
	    	     * transfer data to or from the data FIFO, this is
	    	     * the execution phase
	    	     */

	    	    if (pCmd->command == FDC_READOP) 
	   	        {
	       		x_count = fdcRxdEp(pFdc, bufadd, d_count);
	    		} 
		    else 
			{
	       		x_count = fdcTxdEp(pFdc, bufadd, d_count);
	    		}
	 	    }
      		}

	/*
	 * precondition the results phase status bytes, this is done
       	 * in the event the FDC timeouts while retrieving the status
       	 * data bytes from the results phase
       	 */

	pd_p = (UCHAR *)&cr_data_p->r_data.r_justdata.databytes[0];
      	pd_p[0] = 0xC0 | (pos.headno << 2) | pDev->driveNumber;
      	pd_p[1] = 0x14;
      	pd_p[2] = 0x00;


      	/* read data from data FIFO, results phase */

      	fdcStat(pFdc, pd_p, sizeof(cr_data_p->r_data.r_readdata));

      	/*
      	 * check for errors, if so, set up the additional error
      	 * status data
      	 */

	statusx = pd_p[0];
	if ((statusx & 0xC0) != 0x00) 
	    {
	    switch (statusx & 0xC0) 
		{
	    	case 0x40:
	            statusx = pd_p[1];
	            if (statusx == 0x80) 
			{
		  	statusx = pd_p[2];
		  	if (statusx == 0x00) 
			    {
		     	    break;
		  	    }
	       		}
	    	case 0x80:
	    	case 0xC0:
	       	    lstatus = (UINT)-1;
	            break;
	 	}
	    if (lstatus) 
		{
	    	pCmd->status = ((pCmd->command == FDC_READOP) ?
		 		FDC_ERROR_READ : FDC_ERROR_WRITE);
	        fdcSetAES(pCmd, pd_p, 3);
	        break;
	 	}
	    }

	/*
      	 * verify the expected data count with the actual data count,
      	 * if they do not match, post the appropriate error status
      	 */

      	if (d_count != x_count) 
	    {
	    pCmd->e_count = d_count;
	    pCmd->a_count = x_count;
	    pCmd->status = FDC_ERROR_DATACOUNT;
	    lstatus = (UINT)-1;
	    break;
            }

	/* copy internal buffer to caller's buffer */

      	if (fdcDrvDmaChannel != (UINT)-1) 
	    {
	    if (pCmd->command == FDC_READOP) 
		{
		cacheInvalidate(DATA_CACHE, (void *)pDev->dmaBuffer, d_count);
	        memcpy((void *) bufadd, (void *) pDev->dmaBuffer, d_count);
	        }
      	    }

	/*
      	 * update the logical sector number
      	 * update the accumulated data count
      	 * update the buffer address
      	 */

      	pos.lsector += (d_count / pDev->fdcType.sectorsize);
      	m_count -= d_count;
      	bufadd += d_count;
	}

    /* motor off */

    pFdc->dor &= (FDC_DOR_DMAEN|FDC_DOR_RESET);

    return (lstatus);
    }

/*******************************************************************************
*
* fdcFormat - format track/disk
*
* This function's purpose is to format the specified number
* of tracks (i.e., one or all) of the specified device.  The
* format is specified by the configuration parameters.
*
* RETURNS: 0 if OK, non-zero for error condition which means the status
*  word in the command packet is set.
*/

LOCAL UINT fdcFormat
    (
    register FDC_DEV *pDev,		/* device descriptor pointer */
    register FDC_CMDPCKT *pCmd,		/* command packet pointer */
    register FDC *pFdc,			/* FDC registers pointer */
    register FDC_CRDATA *cr_data_p	/* command/result data pointer */
    )
    {
    register UINT d_count, x_count;	/* data count variables */
    register UINT n_tracks;		/* number of track variable */
    register UINT lstatus;		/* local status flag */
    register UCHAR statusx;		/* status register copy */
    register UCHAR *pd_p;		/* phase data pointer */

    struct fdc_pos pos;

    UCHAR addressfieldbytes[64][4];	/* address field bytes array */

    /* setup the return status */

    lstatus = 0;

    /* setup the number of tracks to format variable */

    if (pCmd->tdflg == 'T') 
	{

        /* calculate logical sector number of selected track */

        pos.lsector =
        fdc_clsn( pCmd->blcknum, pDev->fdcType.sectorsize,
	          pDev->blockSize, pDev->fdcType.sectorstrack, 1 );

        /* take sector number to a track boundary */

        pos.lsector = 
	(pos.lsector / pDev->fdcType.sectorstrack) * pDev->fdcType.sectorstrack;

        /* check the sanity of logical sector number */

        if (pos.lsector >=
	    (pDev->fdcType.sectorstrack * pDev->fdcType.numberofheads * 
             pDev->fdcType.numberoftracks)) 
	    {
	    pCmd->status = FDC_ERROR_ILLDREQ;
	    return ((UINT)-1);
            }

        n_tracks = 1;
	} 
    else 
	{
        pos.lsector = 0;
        n_tracks = pDev->fdcType.numberoftracks * pDev->fdcType.numberofheads;
        }

    /* check state (take to known state) */

    if (fdcCheck(pDev, pCmd, pFdc, cr_data_p, 0)) 
	{
        lstatus = (UINT)-1;
        n_tracks = 0;
        }


    /* format tracks loop */

    for (; n_tracks; n_tracks--, pos.lsector += pDev->fdcType.sectorstrack) 
	{

        /* calculate floppy disk address from logical sector number */

        fdcCP( pDev->fdcType.numberofheads,
	       pDev->fdcType.sectorstrack,
	       (struct fdc_pos *)&pos );

        /*
         * seek to track to format
         *
         * if the seek to track call fails, post error status and
         * terminate for loop
         */

        if (fdcSeek(pDev, pCmd, pFdc, cr_data_p, pos.headno, pos.cylndrno)) 
	    {
	    lstatus = (UINT)-1;
	    break;
            }

        /* setup the "format-track" command structure */

        pd_p = (UCHAR *)&cr_data_p->c_data.c_justdata.databytes[0];
        bzero((UCHAR *)cr_data_p, sizeof(FDC_CRDATA));
        pd_p[0] = FDC_CP_MFM | FDC_CS_FORMATTRACK;
        pd_p[1] = pDev->driveNumber | (pos.headno ? FDC_ST0_HDS : 0);
        pd_p[2] = fdcSCode(pDev->fdcType.sectorsize);
        pd_p[3] = pDev->fdcType.sectorstrack;
        pd_p[4] = (UCHAR)pDev->fdcType.gapformat;
        pd_p[5] = 0xF6;

        /* setup the "format-track" data phase (address field bytes) */

        d_count = pDev->fdcType.sectorstrack;
        for (x_count = 0; x_count < d_count; x_count++) 
	    {
	    addressfieldbytes[x_count][0] = pos.cylndrno;
	    addressfieldbytes[x_count][1] = pos.headno;
	    addressfieldbytes[x_count][2] = pos.sectorno + x_count;
	    addressfieldbytes[x_count][3] = pd_p[2];
	    }
        d_count *= 4;


        /* setup/start the DMA controller */

        if (fdcDrvDmaChannel != (UINT)-1) 
	    {

	    /* copy format-data buffer to internal buffer */

	    memcpy((void *) pDev->dmaBuffer, (UCHAR *)&addressfieldbytes[0], d_count);
	    cacheFlush(DATA_CACHE, (void *)pDev->dmaBuffer, d_count);


	    /* initialize DMA controller for DMA data transfer */

	    isaDmaStart( fdcDrvDmaChannel,
		         I8237_MODE_TM_DEMAND,
		         I8237_MODE_TT_READ,
		         pDev->dmaBuffer, d_count );
	    }


        /* issue the "format-track" command */

        fdcExeCmd(pFdc, pd_p, sizeof(cr_data_p->c_data.c_formattrack));

        /* wait for interrupt, or poll for completion */

        if (fdcDrvIntVector != (UINT)-1) 
	    {
	    if (semTake(pDev->intSemId, pDev->sysClkRate * 2) == ERROR) 
		{
	        pFdc->fifo = 0x00;
	        EIEIO_SYNC;
	        isaDmaStop(fdcDrvDmaChannel);
	        x_count = 0;
		} 
	    else 
		{
	        x_count = d_count;
	        }
	    } 
	else 
	    {

	    /* poll DMA controller for completion (terminal count) */

	    if (fdcDrvDmaChannel != (UINT)-1) 
		{
	        for (x_count = 0;; x_count++) 
		    {

		    /* sleep for a bit (10ms) */

	            fdcDelay(10);

		    /* retrieve DMA controller status */

	            if ((statusx = (UCHAR)isaDmaStatus(fdcDrvDmaChannel)) == 1)
			{
		  	x_count = d_count;
		  	break;
	       		} 
		    else 
			{
		  	if (x_count >= (2000/10)) 
			    {
		     	    pFdc->fifo = 0x00;
		     	    EIEIO_SYNC;
		     	    isaDmaStop(fdcDrvDmaChannel);
		     	    x_count = 0;
		     	    break;
		  	    }
	       	   	}
	    	    }
	 	} 
	    else 
		{

	        /* 
		 * transfer data to the data FIFO, 
		 * this is the execution phase 
		 */

	        d_count = pDev->fdcType.sectorstrack * 4;
	        x_count = 
		    fdcTxdEp(pFdc, (UCHAR *)&addressfieldbytes[0], d_count);
		}
	    }

        /*
         * precondition the results phase status bytes, this is done
         * in the event the FDC timeouts while retrieving the status
         * data bytes from the results phase
         */

        pd_p = (UCHAR *)&cr_data_p->r_data.r_justdata.databytes[0];
        pd_p[0] = 0xC0 | (pos.headno << 2) | pDev->driveNumber;
        pd_p[1] = 0x14;
        pd_p[2] = 0x00;


        /* read data from data FIFO, results phase */

        fdcStat(pFdc, pd_p, sizeof(cr_data_p->r_data.r_formattrack));

        /*
         * check for errors, if so, set up the additional error
         * status data
         */

        statusx = pd_p[0];
        if ((statusx & 0xC0) != 0x00) 
	    {
	    switch (statusx & 0xC0) 
		{
	        case 0x40:
	            statusx = pd_p[1];
	            if (statusx == 0x80) 
			{
		        statusx = pd_p[2];
		        if (statusx == 0x00) 
			    {
		     	    break;
		  	    }
	       		}
	    	case 0x80:
	    	case 0xC0:
	            lstatus = (UINT)-1;
	            break;
	 	}
	    if (lstatus) 
		{
		pCmd->status = FDC_ERROR_FORMAT;
		fdcSetAES(pCmd, pd_p, 3);
		break;
		}
	    }

        /*
         * verify the expected data count with the actual data count,
         * if they do not match, post the appropriate error status
         */

        if (d_count != x_count) 
	    {
	    pCmd->e_count = d_count;
	    pCmd->a_count = x_count;
	    pCmd->status = FDC_ERROR_DATACOUNT;
	    lstatus = (UINT)-1;
	    break;
            }
	}


    /* motor off */

    pFdc->dor &= (FDC_DOR_DMAEN|FDC_DOR_RESET);

    return (lstatus);
    }

/*******************************************************************************
*
* fdcSeek - seek to track
* 
* This function's purpose is to seek to the specified track.
* The case of Head Zero and Track Zero, a recalibrate operation
* will be performed.
* 
* RETURNS: 0 if OK, non-zero for error condition which means the status
*  word in the command packet is set.
*/

LOCAL UINT fdcSeek
    (
    register FDC_DEV *pDev,		/* device descriptor pointer */
    register FDC_CMDPCKT *pCmd,		/* command packet pointer */
    register FDC *pFdc,			/* FDC registers pointer */
    register FDC_CRDATA *cr_data_p,	/* command/result data pointer */
    register UINT headno,		/* head number to seek to */
    register UINT trackno 		/* track number to seek to */
    )
    {
    register UINT lstatus;		/* local status flag */
    register UCHAR statusx;		/* status register copy */
    register UCHAR *pd_p;		/* phase data pointer */

    /* setup the return status */

    lstatus = 0;

    /*
     * if the Head Zero and Track Zero case is true, issue a
     * recalibrate, else issue a seek command
     */

    if ((headno == 0) && (trackno == 0)) {

      /* setup the "recalibrate" command structure */

      pd_p = (UCHAR *)&cr_data_p->c_data.c_justdata.databytes[0];
      bzero((UCHAR *)cr_data_p, sizeof(FDC_CRDATA));
      pd_p[0] = FDC_CS_RECALIBRATE;
      pd_p[1] = pDev->driveNumber;


      /* issue the "recalibrate" command */

      fdcExeCmd(pFdc, pd_p, sizeof(cr_data_p->c_data.c_recalibrate));
    } else {

      /* setup the "seek" command structure */

      pd_p = (UCHAR *)&cr_data_p->c_data.c_justdata.databytes[0];
      bzero((UCHAR *)cr_data_p, sizeof(FDC_CRDATA));
      pd_p[0] = FDC_CS_SEEK;
      pd_p[1] = pDev->driveNumber | (headno ? FDC_ST0_HDS : 0);
      pd_p[2] = trackno;


      /* issue the "seek" command */

      fdcExeCmd(pFdc, pd_p, sizeof(cr_data_p->c_data.c_seek) - 1);
   }


    /* wait for interrupt from the selected drive */

    for (;;) 
	{

        /* delay a bit, wait for interrupt status */

        if (fdcDrvIntVector != (UINT)-1) 
	    {
	    if (semTake(pDev->intSemId, pDev->sysClkRate * 4) == ERROR) 
		{
	        pCmd->status = FDC_ERROR_SEEK;
	        fdcSetAES(pCmd, pd_p, 2);
	        lstatus = (UINT)-1;
	        break;
		}
	    } 
	else 
	    {
	    fdcDelay(200);
	    }

        /* setup the "sense-interrupt" command structure */

        pd_p = (UCHAR *)&cr_data_p->c_data.c_justdata.databytes[0];
        bzero((UCHAR *)cr_data_p, sizeof(FDC_CRDATA));
        pd_p[0] = FDC_CS_SENSEINT;

        /* issue the "sense-interrupt" command */

        fdcExeCmd(pFdc, pd_p, sizeof(cr_data_p->c_data.c_senseinterrupt));

        /* read data (1 byte only) from data FIFO, results phase */

        pd_p = (UCHAR *)&cr_data_p->r_data.r_justdata.databytes[0];
        fdcStat(pFdc, pd_p, 1);

	/*
         * check for invalid command status, this would be the case
         * if no interrupt was pending, if not invalid, read second
         * byte of sense interrupt result data
         */

	statusx = pd_p[0];
	if ((statusx & 0xC0) != 0x80) 
	    {

	    /* read data (1 byte only) from data FIFO, results phase */

	    fdcStat(pFdc, pd_p + 1, 1);

	    /*
	     * check for errors, if so, set up the additional error
	     * status data
	     */

	    if ((statusx & 0x03) == pDev->driveNumber) 
		{
	        if (statusx & 0x20) 
		    {
	            if ((statusx & 0xC0) != 0x00) 
			{
		        pCmd->status = FDC_ERROR_SEEK;
		        fdcSetAES(pCmd, pd_p, 2);
		        lstatus = (UINT)-1;
			}
	            break;
		    }
		}
	    }
	}

    return (lstatus);
    }

/*******************************************************************************
*
* fdcCheck - check state
* 
* This function's purpose is to check the state of the FDC and
* the state of the specified floppy diskette drive.
*
* RETURNS: 0 if OK, non-zero for error condition which means the status
*  word in the command packet is set.
*/

LOCAL UINT fdcCheck
    (
    register FDC_DEV *pDev,		/* device descriptor pointer */
    register FDC_CMDPCKT *pCmd,		/* command packet pointer */
    register FDC *pFdc,			/* FDC registers pointer */
    register FDC_CRDATA *cr_data_p,	/* command/result data pointer */
    register UINT diskChange		/* disk change clear flag */
    )
    {
    register UINT lstatus;		/* local status flag */

    /* setup the return status */

    lstatus = 0;

    for (; !lstatus;) 
	{

        /* if some outstanding status is present, reset the FDC */

        if (pFdc->msr_dsr != FDC_MSR_RQM) 
	    {
	     EIEIO_SYNC;
	     pFdc->dor = 0x00;
	     EIEIO_SYNC;
	    }


        /* deassert S/W reset bit in DOR, all motors off, no DMA */

        if (!(pFdc->dor & FDC_DOR_RESET)) 
	    {
	    EIEIO_SYNC;
	    if (fdcDrvDmaChannel != (UINT)-1) 
		{
	        pFdc->dor = (FDC_DOR_RESET|FDC_DOR_DMAEN);
		} 
	    else 
		{
	        pFdc->dor = (FDC_DOR_RESET);
		}
	    EIEIO_SYNC;


	    /* clear reset state of FDC */

	    if (fdcClearReset(pDev, pCmd, pFdc, cr_data_p)) 
		{
	        lstatus = (UINT)-1;
	        continue;
		}


	    /* initialize DMA controller */

	    if (fdcDrvDmaChannel != (UINT)-1) 
		{
	        isaDmaInit();
		}
	    }

        /*
         * motor on
         * initialize DATA RATE and PRECOMP (%000 specifies default)
         * clear local error status flag
         */

        pFdc->dor |= (pDev->driveNumber | (0x10 << pDev->driveNumber));
        pFdc->msr_dsr = fdcDRCode(pDev);
        pFdc->dir_ccr = fdcDRCode(pDev);

        /*
         * initialize the FDC with the operating parameters of
         * the attached disk
         *
         * if the initialization calls fails, post error status
         */

        if (fdcInit(pDev, pFdc, cr_data_p)) 
	    {
	    lstatus = (UINT)-1;
	    continue;
	    }

        /*
         * query the digital input register for a disk change status, if
         * a disk change has occurred, seek to head zero, track zero
         *
         * if the seek to track call fails, post error status
         */

	if ((pFdc->dir_ccr & FDC_DIR_PS2_DSKCHG) ||
	  (pDev->fdcBlockDev.bd_readyChanged == TRUE)) 
	    {
	    if (fdcSeek(pDev, pCmd, pFdc, cr_data_p, 0, 0)) 
		{
	        lstatus = (UINT)-1;
	        continue;
		}
	    }

	/* clear disk change status (if directed) */

	if (diskChange) 
	    {

	    /* seek to head zero, track zero */

	    if (fdcSeek(pDev, pCmd, pFdc, cr_data_p, 0, 0)) 
		{
	        lstatus = (UINT)-1;
	        continue;
		}


	    /* seek to head zero, track 8 */

	    if (fdcSeek(pDev, pCmd, pFdc, cr_data_p, 0, 8)) 
		{
	        lstatus = (UINT)-1;
	        continue;
		}


	    /* seek to head zero, track zero */

	    if (fdcSeek(pDev, pCmd, pFdc, cr_data_p, 0, 0)) 
		{
	        lstatus = (UINT)-1;
	        continue;
		}
	    }

        break;
	}

    return (lstatus);
    }

/*******************************************************************************
* fdcClearReset - clear reset state
*
* This function's purpose is to clear the 4 internal disk ready
* line change interrupts.  These interrupts are only present
* following a hardware/software reset.
*
* RETURNS: 0 if OK, non-zero for error condition which means the status
*  word in the command packet is set.
*/

LOCAL UINT fdcClearReset
    (
    register FDC_DEV *pDev,		/* device descriptor pointer */
    register FDC_CMDPCKT *pCmd,		/* command packet pointer */
    register FDC *pFdc,			/* FDC registers pointer */
    register FDC_CRDATA *cr_data_p	/* command/result data pointer */
    )
    {
    register UINT lstatus;		/* local status flag */
    register UCHAR statusx;		/* status register copy */
    register UCHAR *pd_p;		/* phase data pointer */
    register UINT driveno;		/* drive number */


    /* setup the return status */

    lstatus = 0;

    /*
     * delay for a bit, the device must enter the idle
     * phase, accessing the device (FDC) will prohibit
     * the idle phase from being entered
     */

    fdcDelay(2000);

    /* wait for interrupt from the selected drive */

    for (driveno = 0; driveno < FDC_NDRIVES;) 
	{

        /* setup the "sense-interrupt" command structure */

        pd_p = (UCHAR *)&cr_data_p->c_data.c_justdata.databytes[0];
        bzero((UCHAR *)cr_data_p, sizeof(FDC_CRDATA));
        pd_p[0] = FDC_CS_SENSEINT;

        /* issue the "sense-interrupt" command */

        fdcExeCmd(pFdc, pd_p, sizeof(cr_data_p->c_data.c_senseinterrupt));

        /* read data (1 byte only) from data FIFO, results phase */

        pd_p = (UCHAR *)&cr_data_p->r_data.r_justdata.databytes[0];
        fdcStat(pFdc, pd_p, 1);

        /*
         * check for invalid command status, this would be the case
         * if no interrupt was pending, if not invalid, read second
         * byte of sense interrupt result data
         */

        statusx = pd_p[0];
        if ((statusx & 0xC0) != 0x80) 
	    {

	    /* read data (1 byte only) from data FIFO, results phase */

	    fdcStat(pFdc, pd_p + 1, 1);

	    /*
	     * check for errors, if so, set up the additional error
	     * status data
	     */

	    if ((statusx & 0x03) != driveno) 
		{
	        lstatus = (UINT)-1;
	        } 
	    else 
		{
	        if ((statusx & 0xC0) != 0xC0) 
		    {
	            lstatus = (UINT)-1;
	    	    }
		}
	    if (lstatus == (UINT)-1) 
		{
	        pCmd->status = FDC_ERROR_CLEARRESET;
	        fdcSetAES(pCmd, pd_p, 2);
	        break;
	        } 
	    else 
		{
		driveno++;
	        }
	    }

        /* delay a bit, wait for interrupt status */

        fdcDelay(200);
	}

    return (lstatus);
    }

/*******************************************************************************
*
* fdcInit - initialize operating parameters
*
* This function's purpose is to initialize the FDC with
* the operating parameters of the attached floppy disk
* drive.  These operating parameters are specified by the
* user's configuration parameters, as well as defaults
* determined by this driver.
*
* RETURNS: OK, non-zero for error condition which means the status
*  word in the command packet is set.
*/

LOCAL UINT fdcInit
    (
    register FDC_DEV *pDev,		/* device descriptor pointer */
    register FDC *pFdc,			/* FDC registers pointer */
    register FDC_CRDATA *cr_data_p 	/* command/result data pointer */
    )
    {

    /*
     * issue the "mode" command
     *
     * fields that are set zero specifies to use the defaults
     * DENSEL set to default (%11)
     * HEAD-SETTLE set to default code (8)
     */

    bzero((UCHAR *)cr_data_p, sizeof(FDC_CRDATA));
    cr_data_p->c_data.c_mode.opcode = FDC_CS_MODE;
    cr_data_p->c_data.c_mode.cbyte1 = FDC_CP_TMR | 0x2;
    cr_data_p->c_data.c_mode.cbyte2 = 0x00;
    cr_data_p->c_data.c_mode.cbyte3 = 0xC8;
    cr_data_p->c_data.c_mode.cbyte4 = 0x00;
#ifdef FDC_ENHANCED
    fdcExeCmd( pFdc,
	     (UCHAR *)&cr_data_p->c_data.c_justdata.databytes[0],
	     sizeof(cr_data_p->c_data.c_mode)
	   );
#endif

    /*
     * issue the "specify" command
     *
     * fields that are set zero specifies to use the defaults
     * STEP-RATE set to maximum
     * MOTOR-OFF-TIME set to maximum
     * MOTOR-ON-TIME set to a code of 8,
     * (256ms for 1mb/500kbs, 512ms for 250kbs)
     */

    bzero((UCHAR *)cr_data_p, sizeof(FDC_CRDATA));
    cr_data_p->c_data.c_specify.opcode = FDC_CS_SPECIFY;
    cr_data_p->c_data.c_specify.cbyte1 = 0x00;	/* step-rate/motor-off-time */
    cr_data_p->c_data.c_specify.cbyte2 = 0x10;	/* motor-on-time */
    if (fdcDrvDmaChannel == (UINT)-1) 
	{
        cr_data_p->c_data.c_specify.cbyte2 |= FDC_CP_DMA;
        }
    fdcExeCmd( pFdc,
	     (UCHAR *)&cr_data_p->c_data.c_justdata.databytes[0],
	     sizeof(cr_data_p->c_data.c_specify)
	   );


    /* issue the "configure" command */

    bzero((UCHAR *)cr_data_p, sizeof(FDC_CRDATA));
    cr_data_p->c_data.c_configure.opcode = FDC_CS_CONFIGURE;
    cr_data_p->c_data.c_configure.cbyte1 = 0x00;
#ifdef FDC_ENHANCED
    cr_data_p->c_data.c_configure.cbyte2 = (FDC_CP_EIS) | 0x0F;
#else
    cr_data_p->c_data.c_configure.cbyte2 = (FDC_CP_EIS);
#endif
    cr_data_p->c_data.c_configure.pretrk = 0;
    fdcExeCmd( pFdc,
	     (UCHAR *)&cr_data_p->c_data.c_justdata.databytes[0],
	     sizeof(cr_data_p->c_data.c_configure)
	   );

    return (OK);
    }

/*******************************************************************************
*
* fdc_clsn - calculate logical sector number
*
* This function's purpose is to calculate the logical sector
* number from the logical block parameters.
*
* RETURNS: -1 on ERROR, or the logical sector number
*/

LOCAL UINT fdc_clsn
    (
    register UINT bno,		/* block number */
    register UINT ssiz,		/* sector size */
    register UINT bsiz,		/* block size */
    register UINT sectrk,	/* sectors per track */
    register UINT equalinall	/* equal in all */
    )
    {
    UINT ls = (UINT)-1;

    if (ssiz == bsiz ) 
        {
        if (equalinall) 
	    {
	    ls = bno;
	    return (ls);
	    }
	else 
	    {
	    ls = bno * 2;
	    if (ls >= sectrk)
	        ls = bno + (sectrk / 2);
	    return (ls);
	    }
	}

    if (ssiz < bsiz ) 
	{
        if (equalinall) 
	    {
	    ls = bno * (bsiz / ssiz);
	    return (ls);
	    } 
	else 
	    {
	    ls = bno * 2 * (bsiz / ssiz);
	    if (ls >= sectrk)
	        ls = ((bsiz / ssiz) * bno) + (sectrk / 2);
	    return (ls);
	    }
	}

    if (ssiz > bsiz ) 
	{
        if (equalinall) 
	    {
	    ls = bno / (ssiz / bsiz);
	    return (ls);
	    } 
	else 
	    {
	    ls = bno / ((ssiz / bsiz) / 2);
	    if (ls >= sectrk)
	        ls = (bno / (ssiz / bsiz)) + (sectrk / 2);
	    return (ls);
	    }
	}
    return (ls);
    }

/*******************************************************************************
*
* fdcCP - calculate position
* 
* This function's purpose is to calculate the position, in other
* words, the physical sector number, the physical head number,
* and the cylinder number.
*
* RETURNS: N/A
*/

LOCAL void fdcCP
    (
    register UINT nheads,		/* number of heads */
    register UINT sectrk,		/* sectors per track */
    register struct fdc_pos *p	 	/* pointer to position template */
    )
    {
    p->cylndrno = (USHORT)(p->lsector / (nheads * sectrk));
    p->headno = (UCHAR)((p->lsector % (nheads * sectrk)) / sectrk);
    p->sectorno = (UCHAR)(p->lsector % sectrk);
    p->sectorno += 1;
    }

/*******************************************************************************
*
* fdcRxd - receive data from FIFO
*
* This function's purpose is to retrieve one byte from the
* data FIFO for command execution status (result phase data).
*
* RETURNS: Data-Byte, or -1 on FIFO timeout.
*/

LOCAL UINT fdcRxd
    (
    register FDC *pFdc		/* FDC registers pointer */
    )
    {
    register UCHAR rdata;	/* read data local variable */
    register UINT acount;	/* access count variable */

    /*
     * poll MSR for data availability, if data is availabilty,
     * read data FIFO and return with read data
     */

    for (acount = 0;;) 
	{
        rdata = pFdc->msr_dsr;
        if ((rdata & (FDC_MSR_RQM|FDC_MSR_DIO|FDC_MSR_NONDMA)) ==
	    (FDC_MSR_RQM|FDC_MSR_DIO)) 
	    {
	    rdata = pFdc->fifo;
	    return (rdata);
	    } 
	else 
	    {
	    if (++acount > FDC_TIMEOUTCOUNT) 
		{
	        return ((UINT)-1);
		}
	    }
	}
    }

/*******************************************************************************
*
* fdcTxd - transmit (send) byte to data FIFO
*
*	This function's purpose is to send one byte to the
*	data FIFO for command execution.
*
* RETURNS: OK, or -1 on FIFO timeout
*/

LOCAL UINT fdcTxd
    (
    register FDC *pFdc,		/* FDC registers pointer */
    register UCHAR wdata	/* write data */
    )
    {
    register UCHAR rdata;	/* read data local variable */
    register UINT acount;	/* access count variable */

    /*
     * poll MSR for data fifo availability, if data fifo is availabilty,
     * write data FIFO and return
     */

    for (acount = 0;;) 
	{
        rdata = pFdc->msr_dsr;
        if ((rdata & (FDC_MSR_RQM|FDC_MSR_DIO|FDC_MSR_NONDMA)) ==
	    (FDC_MSR_RQM)) 
	    {
	    pFdc->fifo = wdata;
	    break;
	    } 
        else 
	    {
	    if (++acount > FDC_TIMEOUTCOUNT) 
		{
	        return ((UINT)-1);
		}
	    }
	}

    return (OK);
    }

/*******************************************************************************
*
* fdcStat - retrieve command execution status (results)
*
* This function's purpose is to retrieve the command execution
* status (results) bytes from the data FIFO.  The status bytes
* are placed into the buffer as specified by the caller's
* arguments.
*
* RETURNS: N/A
*/

LOCAL void fdcStat
    (
    register FDC *pFdc,		/* FDC registers pointer */
    register UCHAR *data_p,		/* data pointer */
    register UINT data_s		/* data size in bytes */
    )
    {
    register UINT rdata;		/* read data local variable */

    for (; data_s; data_s--, data_p++) 
	{
        rdata = fdcRxd(pFdc);
        if (rdata == (UINT)-1) 
	    break;
        *data_p = rdata;
        }
    }

/*******************************************************************************
* fdcExeCmd - execute command
*
* This function's purpose is to execute the specified command as
* specified by the passed data (pointer and size arguments).
*
* RETURNS: N/A
*/

LOCAL void fdcExeCmd
    (
    register FDC *pFdc,		/* FDC registers pointer */
    register UCHAR *data_p,	/* data pointer */
    register UINT data_s	/* data size in bytes */
    )
    {
    register UINT rdata;		/* read data local variable */

    for (; data_s; data_s--, data_p++) 
	{
        rdata = fdcTxd(pFdc, *data_p);
        if (rdata == (UINT)-1) break;
        }
    }

/*******************************************************************************
*
* fdcRxdEp - receive data from FIFO, execution phase
* 
* This function's purpose is to retrieve all data from the
* data FIFO for the command execution phase.  Only the
* number of bytes as specified will be transferred to
* the data buffer.
*
* RETURNS: number of bytes read
*/

LOCAL UINT fdcRxdEp
    (
    register FDC *pFdc,		/* FDC registers pointer */
    register UCHAR *data_p,	/* data pointer */
    register UINT data_s	/* data size */
    )
    {
    register UINT dcount;	/* local data count variable */
    register UINT acount;	/* access count variable */
    register UCHAR rdata;	/* read data local variable */

    /*
     * poll MSR for data availability, if data is availabilty,
     * read data FIFO and write to caller's data buffer, exit
     * if the the FIFO changes state (execution to result)
     */

    for (acount = 0, dcount = 0; dcount < data_s;) 
	{
        rdata = pFdc->msr_dsr;
        EIEIO_SYNC;
        if (rdata & FDC_MSR_RQM) 
	    {
	    if ((rdata & (FDC_MSR_NONDMA|FDC_MSR_DIO)) ==
	        (FDC_MSR_NONDMA|FDC_MSR_DIO)) 
		{
	        rdata = pFdc->fifo;
	        EIEIO_SYNC;
	        *data_p++ = rdata;
	        dcount++;
	        acount = 0;
		} 
	    else 
		{
	        break;
		}
	    } 
        else 
	    {
	    if (++acount > FDC_TIMEOUTCOUNT) 
		{
	        pFdc->fifo = 0x00;
	        break;
		}
	    }
	}

    return (dcount);
    }

/*******************************************************************************
*
* fdcTxdEp - transmit data to FIFO, execution phase
*
* This function's purpose is to read all data from the data
* caller's buffer and write it to the data FIFO.  This is for
* the command execution phase only.  Only the number of bytes
* as specified will be transferred to the data FIFO.
* 
* RETURNS: number of bytes written.
*/

LOCAL UINT fdcTxdEp
    (
    register FDC *pFdc,		/* FDC registers pointer */
    register UCHAR *data_p,	/* data pointer */
    register UINT data_s	/* data size */
    )
    {
    register UINT dcount;	/* local data count variable */
    register UINT acount;	/* access count variable */
    register UCHAR rdata;	/* read data local variable */

    /*
     * poll MSR for data availability, if data is availabilty,
     * read caller's data buffer and write data to FIFO, exit
     * if the the FIFO changes state (execution to result)
     */

    for (acount = 0, dcount = 0; dcount < data_s;) 
	{
        rdata = pFdc->msr_dsr;
        EIEIO_SYNC;
        if (rdata & FDC_MSR_RQM) 
	    {
	    if ((rdata & (FDC_MSR_NONDMA|FDC_MSR_DIO)) ==
	        (FDC_MSR_NONDMA)) 
		{
	    pFdc->fifo = *data_p++;
	    dcount++;
	    EIEIO_SYNC;
	    acount = 0;
		} 
	    else 
		{
	    break;
		}
	    } 
	else 
	    {
	    if (++acount > FDC_TIMEOUTCOUNT) 
		{
	        pFdc->fifo = 0x00;
	        break;
		}
	    }
	}

    return (dcount);
    }

/*******************************************************************************
*
* fdcDRCode - return data-rate code
* 
*This function's purpose is to return the data-rate code.
*This based upon the speed and sectors per track.
*
* RETURNS: data-rate code
*/

LOCAL UCHAR fdcDRCode
    (
    register FDC_DEV *pDev		/* device descriptor pointer */
    )
    {
    register UCHAR datarate;

    switch (pDev->fdcType.ratedata) 
	{
        case 250:
	    datarate = 0x02;
	    break;
        case 500:
	    datarate = 0x00;
	    break;
        case 1000:
	    datarate = 0x03;
	    break;
        default:
	    datarate = 0x00;
	    break;
	}

    return (datarate);
    }

/*******************************************************************************
*
* fdcSCode - return sector code
*
* This function's purpose is to return the sector code based
* upon the physical sector size.
*
* RETURNS: sector code.
*/

LOCAL UINT fdcSCode
    (
    register UINT ssize		/* sector size */
    )
    {
    register UINT multiple, npower;

    multiple = ssize / 128;	/* divide by base code value */

    for (npower = 0;; multiple >>= 1, npower++) 
	{
        if (multiple & 0x1) 
	    {
	    break;
	    }
	}

    return (npower);
    }

/*******************************************************************************
*
* fdcSetAES - setup additional error status information
*
* This function's purpose is to setup the additional error status
* information packet in the mass storage i/o subsystem.  This
* additional error status information is used by the upper layers
* for display/reporting of errors.
*
* RETURNS: N/A
*/

LOCAL void fdcSetAES
    (
    register FDC_CMDPCKT *pCmd,	/* command packet pointer */
    register UCHAR *p,		/* status data pointer */
    register UINT s		/* status data size (i.e., number of bytes) */
    )
    {
    register UINT index;

    pCmd->aescount = s;
    for (index = 0; index < s; pCmd->aesdata[index++] = *p++);
    }

/*******************************************************************************
*
* fdcDelay - delay
*
* This function's purpose is to delay (sleep) for the specified
* number of milli-seconds.
*
* RETURNS: N/A
*/

LOCAL void fdcDelay
    (
    register UINT msDelay	/* number of milli-seconds to delay */
    )
    {
    register UINT msTicks;

    msTicks = 1000 / sysClkRateGet();

    if (msDelay % msTicks) msDelay += msTicks;

    taskDelay(msDelay / msTicks);
    }

/*******************************************************************************
*
* fdcXfrcheck - sanity check on transfer request
*
* This function's purpose is to make a sanity check on the transfer
* request, this is done by examining the logical block size in
* conjunction with the physical block size of the media and the number
* of requested logical blocks.  If the block number denotes a
* physical block number (not a file number), the block number is checked
* for the modulus condition.
*
* RETURNS: OK, or ERROR if sanity check failed
*/

LOCAL UINT fdcXfrcheck
    (
    register FDC_DEV *pDev,		/* device descriptor pointer */
    register FDC_CMDPCKT *pCmd,	/* command packet pointer */
    register UINT bvsf	/* block number verse file number flag, true = file */
    )
    {
    if (!((pCmd->nblcks * pDev->blockSize) / pDev->fdcType.sectorsize)
        || ((pCmd->nblcks * pDev->blockSize) % pDev->fdcType.sectorsize)) 
	{
        pCmd->status = FDC_ERROR_ILLDREQ;
        return ((UINT) -1);
	}
    if (pDev->blockSize < pDev->fdcType.sectorsize) 
	{
	if (!bvsf) 	/* block or file number? */
	    {
	    if (pCmd->blcknum % (pDev->fdcType.sectorsize / pDev->blockSize)) 
		{
		pCmd->status = FDC_ERROR_ILLDREQ;
		return ((UINT) -1);
		}
	    }
	}
    return (OK);
    }

/*******************************************************************************
*
* fdcInt - FDC interrupt handler
*
* This function's purpose is handle/process the FDC interrupt
* request.  The FDC's interrupt request line is active high, and
* is edge triggered.  The interrupt request line is negated
* when the result bytes are read from the FDC.  This interrupt
* handler basically returns control to the task that is blocked
* on the interrupt semaphore (taking ownership).
*
* RETURNS: N/A
*/

LOCAL void fdcInt
    (
    register FDC_IARGS *pFdcIArgs
    )
    {

    /* increment global interrupt counter */

    fdcDrvIntCount[pFdcIArgs->pDev->driveNumber] += 1;


    /* return to task level */

    if (fdcDrvIntVector != (UINT)-1) 
	{
	semGive(pFdcIArgs->pDev->intSemId);
	}
    }

#endif /* INCLUDE_FD */
