/*******************************************************************************
*
* "@(#) $Id: rtLog.c,v 1.8 2009/10/07 09:29:21 bjeram Exp $"
*
* ALMA - Atacama Large Millimiter Array
* (c) Associated Universities Inc., 2003 
*
*This library is free software; you can redistribute it and/or
*modify it under the terms of the GNU Lesser General Public
*License as published by the Free Software Foundation; either
*version 2.1 of the License, or (at your option) any later version.
*
*This library is distributed in the hope that it will be useful,
*but WITHOUT ANY WARRANTY; without even the implied warranty of
*MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
*Lesser General Public License for more details.
*
*You should have received a copy of the GNU Lesser General Public
*License along with this library; if not, write to the Free Software
*Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*
* who       when        what
* --------  --------    ----------------------------------------------
*/

static char *rcsId="@(#) $Id: rtLog.c,v 1.8 2009/10/07 09:29:21 bjeram Exp $";

/*
 * System stuff
 */
#include <linux/moduleparam.h>
#include <linux/delay.h>
#include <asm/atomic.h>
#include <linux/time.h>

/*
 * RTAI stuff
 */
#include <rtai_lxrt.h>
#include <rtai_registry.h>
#include <rtai_sem.h>
#include <rtai_fifos.h>

/*
 * Local stuff
 */
#include <rtLog.h>


MODULE_AUTHOR("Bogdan Jeram");
MODULE_DESCRIPTION("ACS RT LOG");
MODULE_LICENSE("GPL");

/*
 * used for logging
 */
#define modName    "rtLog"

/*
 * RTOS kernel module stages. This value decides, what the main
 * function does. (Copied from rtTools)
 */
#define RT_LOG_MODULE_STAGE_INIT 1
#define RT_LOG_MODULE_STAGE_EXIT 2
#define RT_LOG_MODULE_STAGE_CLEANUP 3

/*
 * RTOS kernel module error codes.
 */
#define RT_LOG_MODULE_INIT_SUCCESS 0
#define RT_LOG_MODULE_EXIT_SUCCESS 0

/*
 * Define an error base which does not conflict with the Linux and
 * kernel errnos.
 */
#define RT_LOG_MODULE_ERROR_BASE 256
/*
 * Standard errors.
 */
#define RT_LOG_MODULE_INIT_ERROR -(RT_LOG_MODULE_ERROR_BASE + 1)
#define RT_LOG_MODULE_CLEANUP_ERROR -(RT_LOG_MODULE_ERROR_BASE + 3)



/*
 * module parameters
 */
static unsigned int initTimeout = 1000;         /* msec */
static unsigned int cleanUpTimeout = 1000;      /* msec */
module_param(initTimeout, uint, S_IRUGO);
module_param(cleanUpTimeout, uint, S_IRUGO);

/*
 * my local data
 */

atomic_t rtlog_level = ATOMIC_INIT(RTLOG_TRACE_LEVEL);
rtLogTimeData_t rtLog_time_data = {0, 0, 0, 0}; /* time data */
SEM time_data_sem;    /* semaphore to protect time data */

/*
 * API
 */
/*
 * getRTLogLevel
 * returns currnet RTlog level
*/
RTAI_SYSCALL_MODE RTLogLevelType getRTLogLevel( void ){ return atomic_read(&rtlog_level); }

/*
 * setRTLogLevel
 * sets currnet RTlog level
*/
RTAI_SYSCALL_MODE void setRTLogLevel( RTLogLevelType level ) 
{ 
    rtlogRecord_t		logRecord;

    atomic_set(&rtlog_level, level); 
    RTLOG_INFO(modName, "rtlog_levle set to: %d (%d)", atomic_read(&rtlog_level), level);
}

RTAI_SYSCALL_MODE int getRTLogTimeData (rtLogTimeData_t* td)
{
    rtlogRecord_t logRecord;
    int stat = rt_sem_wait_timed(&time_data_sem, 1000000);

    if ( stat == 0xFFFF || stat == 0xFFFE )
	{
	RTLOG_ERROR(modName,
	      "failed to take access semaphore in getRTLogTimeData (stat=%d)!", stat);
	return -1;
	}
    *td = rtLog_time_data;

    rt_sem_signal (&time_data_sem);
    
    return 0;
}


/*
 * exported API for LXRT
 */
static struct rt_fun_entry rtlogHandlerFun[] = 
{
    [ RTLOG_SET_LOG_LEVEL ] = { 1, setRTLogLevel },
    [ RTLOG_GET_LOG_LEVEL ] = { 1, getRTLogLevel },
    [ RTLOG_GET_TIME_DATA ] = { UW1(1,2), getRTLogTimeData },
};

EXPORT_SYMBOL (rtlog_level);
EXPORT_SYMBOL (getRTLogLevel);
EXPORT_SYMBOL (setRTLogLevel);

EXPORT_SYMBOL (setRtLogTimeData);

/**************************************************
 *
 * setRtLogTimeData
 *
 *****************************************************/

void setRtLogTimeData(rtLogTimeData_t td)
{
    rtlogRecord_t logRecord;

    int stat = rt_sem_wait_if(&time_data_sem);

    if ( stat == 0xFFFF )
    {
        RTLOG(modName, RTLOG_ERROR, "too bad that time_data_sem is invalid!");
        BUG();
	return;
    }
    else if ( stat == 0 )
    {
        RTLOG(modName, RTLOG_WARNING, "time_data_sem not available");
	return;
    }
    else
    {
        rtLog_time_data = td;
    }

    rt_sem_signal (&time_data_sem);
} /* setRtLogTimeData */

/*****************************************************
 *
 *   rtlog_main
 *
 ******************************************************/
static int rtlog_main(int stage)
{
    int status;

    int cleanUpStatus = RT_LOG_MODULE_EXIT_SUCCESS;
    rtlogRecord_t logRecord;
    int retValue;
    int rtfDestroyCount;

    if ( stage == RT_LOG_MODULE_STAGE_INIT )
    {
        status = RT_LOG_MODULE_INIT_ERROR;

        /*
         * log RCS ID (cvs version)
         */
        RTLOG_INFO(modName, "%s", rcsId);

        RTLOG_INFO(modName, "initializing module...");

        RTLOG_INFO(modName, "initTimeout    = %dms", initTimeout);
        RTLOG_INFO(modName, "cleanUpTimeout = %dms", cleanUpTimeout);

        goto Initialization;
    }
    else
    {
        status = RT_LOG_MODULE_EXIT_SUCCESS;

        RTLOG_INFO(modName, "Cleaning up module...");

        goto FullCleanUp;
    }

Initialization:
    /* export functions for lxrt */
    if ( ( retValue = set_rt_fun_ext_index(rtlogHandlerFun, RTLOG_IDX) ) )
	{
	RTLOG_ERROR(modName,
		    "Recompile %s module with different RTLOG_IDX (=%d) (err=%d).", modName, RTLOG_IDX, retValue);
	status = RT_LOG_MODULE_INIT_ERROR;
	goto Exit;
	}//if

     /* create FIFO for sending logs to the user space */
    if ( (retValue = rtf_create((1), sizeof(RTLogLevelType))) < 0 )
	{
	RTLOG_ERROR(modName, "Cannot create T0 fifo (err=%d).", retValue);
	
	status = RT_LOG_MODULE_INIT_ERROR;
        goto Level1CleanUp;
	}//if
    
    /* initialize semaphore     */
    rt_typed_sem_init(&time_data_sem, 1, RES_SEM);

    /*  initialization is finished    */
    status = RT_LOG_MODULE_INIT_SUCCESS;
    goto Exit;
    
FullCleanUp:

    /* delete semaphore object  */
     rt_sem_delete(&time_data_sem);

    /* destoying fifo */
    /* do this only if we get extra open handles for some reason. */
     do
	 {
	 rtfDestroyCount = rtf_destroy((1));
	 }
     while( rtfDestroyCount > 0 );

     if ( rtfDestroyCount >= 0 )
	 {
	 cleanUpStatus = RT_LOG_MODULE_EXIT_SUCCESS;
	 }
     else
	 {
	 RTLOG_ERROR(modName, "Cannot destroy T0 fifo (err=%d).", rtfDestroyCount);
	 cleanUpStatus = RT_LOG_MODULE_CLEANUP_ERROR;
	 }//if

Level1CleanUp:
     /* clen up funxtions registred for lxrt */
    reset_rt_fun_ext_index(rtlogHandlerFun, RTLOG_IDX);
    cleanUpStatus = RT_LOG_MODULE_EXIT_SUCCESS;

Exit:
    if(stage != RT_LOG_MODULE_STAGE_INIT)
    {
        status = cleanUpStatus;
    }

    return status;
}//rtlog_main

static int __init rtlog_init(void)
{
    int status;
    rtlogRecord_t logRecord;

    if ( (status = rtlog_main(RT_LOG_MODULE_STAGE_INIT))
	 == RT_LOG_MODULE_INIT_SUCCESS )
    {
        RTLOG_INFO(modName, "Module initialized successfully.");
    }
    else
    {
        RTLOG_INFO(modName, "Failed to initialize module!");
    }

    return status;
}//rtlog_init

static void __exit rtlog_exit(void)
{
    rtlogRecord_t logRecord;

    if ( rtlog_main(RT_LOG_MODULE_STAGE_EXIT)
	 == RT_LOG_MODULE_EXIT_SUCCESS )
    {
        RTLOG_INFO(modName, "Module cleaned up successfully.");
    }
    else
    {
        RTLOG_ERROR(modName, "Failed to clean up module!");
    }
}//rtlog_exit

module_init(rtlog_init);
module_exit(rtlog_exit);

/*___oOo___*/
