/*
* Copyright (C) The Community OpenORB Project. All rights reserved.
*
* This software is published under the terms of The OpenORB Community Software
* License version 1.0, a copy of which has been included with this distribution
* in the LICENSE.txt file.
*/
package org.openorb.iiop;

import java.lang.reflect.Constructor;
import java.net.InetAddress;

import org.apache.avalon.framework.CascadingRuntimeException;
import org.apache.avalon.framework.logger.Logger;
import org.apache.avalon.framework.logger.AbstractLogEnabled;

import org.omg.PortableInterceptor.ORBInitInfo;

import org.openorb.net.Address;
import org.openorb.net.TransportClientInitializer;
import org.openorb.PI.FeatureInitInfo;
import org.openorb.util.NumberCache;

/**
 * Interface for creating sockets.
 *
 * @author Chris Wood
 * @version $Revision: 1.10 $ $Date: 2002/09/15 21:48:46 $
 */
public class IIOPTransportClientInitializer extends AbstractLogEnabled
        implements TransportClientInitializer
{
    private static final Constructor IIOP_CTOR;

    private final boolean m_socketNoDelay;
    private final boolean m_boostRecvPriority;
    private final int m_sendBufferSize;
    private final int m_receiveBufferSize;
    private final int m_bufferedOutputStreamSize;
    
    static
    {

        try
        {
            IIOP_CTOR = IIOPTransport.class.getConstructor(
                    new Class [] { InetAddress.class, Integer.TYPE,
                    Logger.class, Boolean.TYPE, Boolean.TYPE, 
                    Integer.TYPE, Integer.TYPE, Integer.TYPE  } );
        }
        catch ( Exception ex )
        {
            throw new CascadingRuntimeException(
                     "Exception during the construction of class IIOPTransport.", ex );
        }
    }

    /**
     * Constructor.
     */
    public IIOPTransportClientInitializer( ORBInitInfo orbinfo, FeatureInitInfo featureinfo )
    {
        final org.openorb.CORBA.kernel.ORBLoader loader = featureinfo.getLoader();

        m_socketNoDelay = 
                loader.getBooleanProperty( "iiop.clientNoDelay", true );
        
        m_boostRecvPriority = 
                loader.getBooleanProperty( "iiop.boostReceivePriority", true );

        m_sendBufferSize = 
                loader.getIntProperty( "iiop.sendBufferSize", 0 );

        m_receiveBufferSize = 
                loader.getIntProperty( "iiop.receiveBufferSize", 0 );
        
        m_bufferedOutputStreamSize = 
                loader.getIntProperty( "iiop.bufferedOutputStreamSize", 50 );
    }

    /**
     * Set the MessageTransport constructor for each of the addresses.
     * All the addresses will be alternative endpoints from a single IOR profile.
     *
     * @return new list of addresses.
     */
    public Address[] establishTransports( final Address[] addresses )
    {
        for ( int i = 0; i < addresses.length; i++ )
        {
            final IIOPAddress addr = ( IIOPAddress ) addresses[ i ];
            
            final Object [] args = new Object[] { addr.get_host(),
                    NumberCache.getInteger( addr.get_port() ), getLogger(),
                    m_socketNoDelay ? Boolean.TRUE : Boolean.FALSE, 
                    m_boostRecvPriority ? Boolean.TRUE : Boolean.FALSE,
                    NumberCache.getInteger( m_sendBufferSize ),
                    NumberCache.getInteger( m_receiveBufferSize ),
                    NumberCache.getInteger( m_bufferedOutputStreamSize ) };
                    
            addresses[ i ].setTransportConstructor( IIOP_CTOR, args );
        }

        return addresses;
    }
}

