-------------------------------------------------------
-- Design Name : tx_ctrl_8b10b 
-- File Name   : tx_ctrl_8b10b.vhd
-- Function    : controlling the 8b10b transmition
-- Coder       : K.-H. Sulanke, DESY
-- Date        : 2019-12-19
-------------------------------------------------------
-- 2019-12-13, half duplex arbitration implemented
-- 2019-12-19, MAX_PACKET_SIZE increased to 16KB


library ieee;
    use ieee.std_logic_1164.all;
    use ieee.std_logic_unsigned.all;

entity tx_ctrl_8b10b is
   port(
        com_reset          : in  std_logic;   -- communication com_reset
        com_clk            : in  std_logic;   -- communication clock
        ena_tx             : in  std_logic;   -- 
        tx_done            : out std_logic;   -- handshake with ena_tx
        tx_fifo_empty      : in  std_logic;   -- 
        tx_fifo_rd_en      : out std_logic;   -- pulse of one clock length
        tx_fifo_dout       : in  std_logic_vector (7 downto 0);
        STF_COMMA          : in  std_logic_vector (7 downto 0); -- start of frame comma
        EOF_COMMA          : in  std_logic_vector (7 downto 0); -- end of frame comma
        enc_8b10b_reset    : out std_logic;  -- 
        enc_8b10b_ena      : out std_logic;  --
        enc_8b10b_in       : out std_logic_vector (7 downto 0); --
        enc_comma_stb      : out std_logic;  -- encoder input is a comma operator
        tx_run_8b10b       : out std_logic;  -- start 8b10 transmitter         
        tx_ack_8b10b       : in  std_logic;  -- one clock length ready pulse
        tx_done_8b10b      : in  std_logic   -- 
       );
end entity;

architecture tx_ctrl_8b10b_arch of tx_ctrl_8b10b is

  constant FRAME_LENGTH    : natural :=     1; -- amount of STFs and EOFs per packet
  constant MAX_PACKET_SIZE : natural := 32768; -- max. amount of data bytes per packet
  constant TX_DONE_DELAY   : natural :=    31; -- amount of clocks after EOF before tx_done <= '1'
  constant D21_5           : std_logic_vector (7 downto 0) := B"101_10101"; --  => 8b10-code(a..j) = B"101010_1010", LEVEL_BALANCER
  
  signal tx_cnt        : natural range 0 to MAX_PACKET_SIZE;
  signal stf           : std_logic_vector (7 downto 0); 
  signal eof           : std_logic_vector (7 downto 0);
  signal wait_ct       : natural range 0 to TX_DONE_DELAY; 
  
type state_type is (TX_IDLE,  TX_BAL,  TX_STF1, TX_FRD, TX_DATA, TX_DATA_ENC_ENA,  TX_EOF1, TX_WAIT, TX_READY);
	signal state: state_type := TX_IDLE;
 
 begin
 
   stf <= STF_COMMA;
   eof <= EOF_COMMA;
                          
   tx_sm: process (com_clk)
	   begin
	  sync: if (rising_edge(com_clk)) then
	           
     reset_if: if (com_reset = '1') then
            state           <= TX_IDLE;
            enc_8b10b_ena   <= '0'; 
            tx_run_8b10b    <= '0';
            enc_8b10b_in    <= X"00";
            enc_comma_stb   <= '0';
            enc_8b10b_reset <= '1';
            tx_fifo_rd_en   <= '0';
            tx_done         <= '0';
         else
			  			  		
--		sm_ena : if (tx_ena ='1') then
			sm: case (state) is
         when TX_IDLE =>
            tx_run_8b10b    <= '0';
            enc_8b10b_reset <= '0';
            enc_8b10b_ena   <= '0'; 
            enc_8b10b_in    <= X"00";
            enc_comma_stb   <= '0'; 
            tx_cnt          <= 0; 
            tx_fifo_rd_en   <= '0';
            tx_done         <= '0';
			if ena_tx = '1' then      --(tx_fifo_empty ='0') then
   			 state           <= TX_BAL; 
   			 enc_8b10b_in    <= D21_5; -- balance the line by 8b10b code: 1010101010
   			 enc_8b10b_ena   <= '1';
			end if;
                
         when TX_BAL =>  
            enc_8b10b_ena   <= '0';
            tx_run_8b10b    <= '1';                    
            if tx_ack_8b10b = '1' then
              tx_run_8b10b  <= '0';
              enc_8b10b_in  <= stf;
              enc_comma_stb <= '1';
              enc_8b10b_ena <= '1';            
  			  state         <= TX_STF1;
            end if;
       
         when TX_STF1 =>
            enc_8b10b_ena  <= '0';
            enc_comma_stb  <= '0';
            tx_run_8b10b   <= '1';
            if tx_ack_8b10b = '1' then
             tx_run_8b10b  <= '0';
             if (tx_fifo_empty ='0') then
     	      state        <= TX_DATA_ENC_ENA; -- fifo with first word fall thru
     	     else
              enc_8b10b_in  <= eof;
              enc_comma_stb <= '1';
              enc_8b10b_ena <= '1';                 
  			  state         <= TX_EOF1;
  			 end if; 
            end if;

        when TX_FRD =>
             tx_fifo_rd_en <= '0';
             --enc_8b10b_ena <= '1';          
       	     state         <= TX_DATA_ENC_ENA;--TX_DATA;

         when TX_DATA_ENC_ENA =>
             --tx_fifo_rd_en <= '0';
             enc_8b10b_in  <= tx_fifo_dout;
             enc_8b10b_ena <= '1';          
       	     state         <= TX_DATA;
            
         when TX_DATA =>
            enc_8b10b_ena   <= '0';
            if (tx_cnt = MAX_PACKET_SIZE) or (tx_fifo_empty = '1') then
               --tx_cnt        <= 0;
               enc_8b10b_in  <= eof;
               enc_comma_stb <= '1';
               enc_8b10b_ena <= '1';                 
  			   state         <= TX_EOF1;
            else         
              tx_run_8b10b    <= '1';                
              if tx_ack_8b10b = '1' then
               tx_run_8b10b   <= '0';
               tx_fifo_rd_en  <= '1';
               state          <= TX_FRD;--TX_DATA_ENC_ENA;
              end if; 
             end if;

         when TX_EOF1 =>
                enc_8b10b_ena  <= '0';
                enc_comma_stb  <= '0';
                tx_run_8b10b   <= '1';
                if tx_ack_8b10b = '1' then
                 tx_run_8b10b  <= '0'; 
--                end if;
--                if tx_done_8b10b = '1' then
                 wait_ct       <= TX_DONE_DELAY;
 	 			 state         <= TX_WAIT;
                end if;
                
          when TX_WAIT =>
                if wait_ct = 0 then
                 state         <= TX_READY;
                else
                 wait_ct <= wait_ct -1; 
                end if;                 
                
          when TX_READY =>
               tx_done        <= '1';
               if ena_tx = '0' then
                 tx_done       <= '0';
                 state         <= TX_IDLE;
                end if;                 
--          when TX_READY =>
--                if tx_done_8b10b = '1' then
--                 tx_done        <= '1';
--                elsif ena_tx = '0' then
--                 tx_done       <= '0';
--                 state         <= TX_IDLE;
--                end if;              
 
        end case sm;
        
   end if reset_if;
  end if sync;
 end process tx_sm;
 
end architecture tx_ctrl_8b10b_arch;
