-------------------------------------------------------
-- Design Name      : icm_top_009 
-- File Name        : icm_top_009.vhd
-- Device           : Spartan 7, XC7S25FTGB196-2
-- Migration Device : Spartan 7, XC7SxxFTGB196-2
-- Function         : ICM register access module, top level design,
--                    
-- Coder            : K.-H. Sulanke, DESY, 2019-12-20
-------------------------------------------------------
-- half duplex comm., UART-USB conn. replaced by UART-FPGA conn.
--
--P1V8_ISO        FPGA_UART_RTS -- output in static high state with 16mA
--current drive, used as power supply (trial)
--ISO_TXD FPGA_UART_TX
--ISO_RXD FPGA_UART_RX
--FPGA_UART_CTS used to switch between the USB2_UART and the ISO_UART
-- (input with internal pullup) will be pulled to GND by the iso-adapter.
-- 2019-12-19, rev. 008, receiver_8b10b bug fixed, tx buffer increased to 32KB
-- 2019-12-20, rev. 009, connection changed, FPGA_UART_RX/TX -> STM_USART_RX/TX


library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_unsigned.all;
use ieee.numeric_bit.all;
use ieee.std_logic_arith.all;

library unisim ;
   use unisim.vcomponents.all ;

entity icm_top_009 is 
port(    

      
      -- ************************ Bank #14, VCCO = 1.8V  ****************************************************
      FPGA_CLK_P    : out std_logic;  -- DIFF_SSTL18_I, clock, delivered to the mainboard
      FPGA_CLK_N    : out std_logic;  --
      FPGA_CLK      : out std_logic;  -- LVCMOS18, copy of the differential clock, delivered to the mainboard
      FPGA_CLK_FBCK : in  std_logic;  -- LVCMOS18, FPGA_CLK feeded back 
      FPGA_SNC_P    : out std_logic;  -- DIFF_SSTL18_I, sync pulse, delivered to the mainboard
      FPGA_SNC_N    : out std_logic;  --
      
      FPGA_GPIO_0   : out std_logic;  -- general purpose IO     
      FPGA_GPIO_1   : out std_logic;  --     
      MCU_nRST      : out std_logic;  --     
      MCU_USART_CK  : out std_logic;  --
          
      FPGA_UART_TX  : in  std_logic;  --  driven by mainboard FPGA   
      FPGA_UART_RX  : out std_logic;  --  received by mainboard FPGA     
      FPGA_UART_CTS : out std_logic;  --  Clear To Send Control Input / Handshake Signal.
      FPGA_UART_RTS : in  std_logic;  --  Request to Send Control Output / Handshake Signal
      
      STM_USART_TX   : in  std_logic;  --  driven by mainboard MCU   
      STM_USART_RX   : out std_logic;  --  received by mainboard MCU     
      STM_USART_CTS  : out std_logic;  -- 
      STM_USART_RTS  : in  std_logic;  --
       
      STM_BOOT      : out std_logic;  --
                                   
      COM_ADC_D     : in  std_logic_vector(13 downto 0);    -- communication ADC output
      COM_ADC_CSBn  : out std_logic; --
      COM_ADC_SCLK  : out std_logic; --
      COM_ADC_SDIO  : inout std_logic; -- is inout
      COM_ADC_CLK_P : out std_logic;  -- DIFF_SSTL18_I, comm. ADC  clock     
      COM_ADC_CLK_N : out std_logic;  -- 
      
      D00_MOSI      : inout std_logic;  -- 
      D01_DIN       : inout std_logic;  -- 
      D02           : inout std_logic;  -- 
      D03           : inout std_logic;  -- 
      FLASH_CSn     : out std_logic;  -- 
      
  
      -- ************************ Bank #34, VCCO = 3.3V  ****************************************************
      QOSC_CLK      : in  std_logic;  -- 20 MHz local oscillator
      QOSC_SDA      : out std_logic;  -- I2C DAC for frequency tuning
      QOSC_SCL      : out std_logic;  --
      
      nLED_ENA      : out std_logic;  --
      nRX_LED       : out std_logic;  --
      nTX_LED       : out std_logic;  --
      
      COM_DAC_DB    : out std_logic_vector(11 downto 0);    -- connected to communication DAC
      COM_DAC_CLOCK : out std_logic;  -- 
      COM_DAC_SLEEP : out std_logic;  -- 
           
      USB_TXD       : in  std_logic;  --   
      USB_RXD       : out std_logic;  --       
      USB_CTSn      : out std_logic;  -- 
      USB_RTSn      : in  std_logic;  --
      USB_PWRENn    : in  std_logic;  --
      USB_RESETn    : out std_logic;  --
      
      MBPWR_EN     : out std_logic;  -- 
      INTLK_0       : out std_logic;  -- 
      INTLK_1       : out std_logic;  -- 
      INTLK_2       : out std_logic;  -- 
      INTLK_3       : out std_logic;  -- 

      CAL_TRG_P     : out std_logic; -- 
      CAL_TRG_N     : out std_logic;
      CAL_TIM_P     : out std_logic;
      CAL_TIM_N     : out std_logic;      
      
      MB_ID         : out std_logic; --inout 
      ICM_ID        : out std_logic; --inout
      
      MB_ADR0       : out std_logic;    -- 
      MB_ADR1       : out std_logic;    -- 
      MB_ADR2       : out std_logic;    -- 
 
      TEST_IO_0     : out std_logic;
      TEST_IO_1     : out std_logic;
      TEST_IO_2     : out std_logic;
      TEST_IO_3     : out std_logic;
      
      TERM_FPGA     : out std_logic  -- '1' activates cable termination       
     );
end icm_top_009 ;

architecture arch_icm_top_009 of icm_top_009 is
 
  
 component icm_pll_01 IS
   PORT
   (
      qosc_clk    : IN  STD_LOGIC; --  20 MHz local clock oscillator
      sys_clk     : OUT STD_LOGIC; --  60 MHz, system clock, made by pll_clkout2
      mb_clkx2    : OUT STD_LOGIC; --  50 MHz, system clock, made by pll_clkout2  
      com_clk     : OUT STD_LOGIC; --  60 Mhz, communication clock made by pll_clkout4
      not_com_clk : OUT STD_LOGIC; --  60 Mhz, communication clock made by pll_clkout5, 180 phase shifted
      com_reset   : OUT STD_LOGIC; --  communication reset, synchronous to com_clk
      sys_reset   : OUT STD_LOGIC;  --  synchronous to the system clock
      mcu_reset   : OUT STD_LOGIC  -- 
    );
  end component icm_pll_01;  
 
  component uart_baudrate_generator is
  generic 
       (
         CLK_FREQUENCY : natural := 60_000_000;
         BAUD_RATE     : natural :=  3_000_000  --115_200;--3_000_000;-- 256_000, 2_000_000
        );
   port ( 
         reset    : in  std_logic;
         clk      : in  std_logic;
         tx_ena   : out std_logic;
         rx_ena   : out std_logic
        );
  end component uart_baudrate_generator;
    
  component uart_receiver is
   port(
    reset           : in  std_logic;
    clk             : in  std_logic;
    rx_ena          : in  std_logic;        -- single clock length pulse from baudrate generator
    rx_in           : in  std_logic;        -- serial data on the line
    rx_data_out     : out std_logic_vector (7 downto 0);  -- parallel out
    rx_data_valid   : out std_logic         -- single clock length pulse
    );
  end component uart_receiver;  
 
  component uart_transmitter is
   port(
        reset          :in  std_logic;
        clk            :in  std_logic;
        tx_data_valid  :in  std_logic; -- like tx_fifo_not_empty
        tx_data_in     :in  std_logic_vector (7 downto 0);
        tx_ena         :in  std_logic;
        tx_ack         :out std_logic; -- single clock pulse
		tx_out         :out std_logic  -- serial data out
       );
  end component uart_transmitter;


--component fifo_4kx8_dual_clk_02 IS
--  PORT (
--    rst         : IN STD_LOGIC;
--    wr_clk      : IN STD_LOGIC;
--    rd_clk      : IN STD_LOGIC;
--    din         : IN STD_LOGIC_VECTOR(7 DOWNTO 0);
--    wr_en       : IN STD_LOGIC;
--    rd_en       : IN STD_LOGIC;
--    dout        : OUT STD_LOGIC_VECTOR(7 DOWNTO 0);
--    full        : OUT STD_LOGIC;
--    empty       : OUT STD_LOGIC;
--    wr_rst_busy : OUT STD_LOGIC;
--    rd_rst_busy : OUT STD_LOGIC
--  );
--END component fifo_4kx8_dual_clk_02;

component fifo_32kx8_dclk_pfull is
  Port ( 
    rst : in STD_LOGIC;
    wr_clk : in STD_LOGIC;
    rd_clk : in STD_LOGIC;
    din : in STD_LOGIC_VECTOR ( 7 downto 0 );
    wr_en : in STD_LOGIC;
    rd_en : in STD_LOGIC;
    dout : out STD_LOGIC_VECTOR ( 7 downto 0 );
    full : out STD_LOGIC;
    empty : out STD_LOGIC;
    prog_full : out STD_LOGIC
  );
end component fifo_32kx8_dclk_pfull;

  component enc_8b10b is  
    port(
		reset    : in std_logic ;	                -- Global asynchronous reset (active high) 
		clk      : in std_logic ;	                -- synchronous  byte clock
		ena      : in std_logic ;                   -- clock enable
		data_in  : in std_logic_vector(7 downto 0);
		komma_in : in std_logic ;	                -- komma operator requested
        data_out : out std_logic_vector(9 downto 0) -- synchronous encoded out
	    );
  end component enc_8b10b;
 
  component cable_baudrate_generator is
  port ( 
         reset          : in  std_logic;
         clk            : in  std_logic;
         baudrate_adj   : in std_logic_vector(1 downto 0);
         tx_ena         : out std_logic;
		 rx_ena         : out std_logic
       );
  end component cable_baudrate_generator;
 
  component tx_ctrl_8b10b is port
  (
        com_reset          : in  std_logic;   -- communication com_reset
        com_clk            : in  std_logic;   -- communication clock
        ena_tx             : in  std_logic;   -- 
        tx_done            : out std_logic;   -- handshake with ena_tx
        tx_fifo_empty      : in  std_logic;   -- 
        tx_fifo_rd_en      : out std_logic;   -- pulse of one clock length
        tx_fifo_dout       : in  std_logic_vector (7 downto 0);
        STF_COMMA          : in  std_logic_vector (7 downto 0); -- start of frame comma
        EOF_COMMA          : in  std_logic_vector (7 downto 0); -- end of frame comma
        enc_8b10b_reset    : out std_logic;  -- 
        enc_8b10b_ena      : out std_logic;  --
        enc_8b10b_in       : out std_logic_vector (7 downto 0); --
        enc_comma_stb      : out std_logic;  -- encoder input is a comma operator
        tx_run_8b10b       : out std_logic;  -- start 8b10 transmitter         
        tx_ack_8b10b       : in  std_logic;  -- one clock length ready pulse
        tx_done_8b10b      : in  std_logic   -- 
   );
  end component tx_ctrl_8b10b;
  
  component transmitter_8b10b is port
   (
        com_reset        : in  std_logic;
        com_clk          : in  std_logic;
        tx_run_8b10b     : in  std_logic; -- like tx_fifo_not_empty
        tx_data_in_8b10b : in  std_logic_vector (9 downto 0);
        tx_ena_8b10b     : in  std_logic; -- from baudrate generator
        tx_ack_8b10b     : out std_logic; -- single clock pulse
        tx_done_8b10b    : out std_logic; --      
        tx_quiet_8b10b   : out std_logic; -- when in IDLE state, to control the com_dac
		tx_out_8b10b     : out std_logic  -- serial data out
    );
  end component transmitter_8b10b;

  component com_dac_enc is
   port(
        reset           : in  std_logic;
        clk             : in  std_logic;
		baudrate_adj    : in std_logic_vector(1 downto 0);
        sdin            : in  std_logic; -- serial data input
        com_dac_quiet   : in  std_logic; -- get zero baseline value, when comm. is inactive
        com_dac_in      : out std_logic_vector (11 downto 0); 
        com_dac_clock   : out std_logic
       );
  end component com_dac_enc;

  component com_adc_dec is
   port(
         reset           : in  std_logic;
         clk             : in  std_logic; -- 60 MHz comm. clock
		 rx_ena          : in  std_logic;
		 baudrate_adj    : in std_logic_vector(1 downto 0);
         com_thr_adj     : in  std_logic_vector (1 downto 0); -- set the comm. threshold
         COM_ADC_CSBn    : out std_logic;   --
         COM_ADC_SCLK    : out std_logic;   --
         COM_ADC_SDIO    : inout std_logic; --
         COM_ADC_D       : in  std_logic_vector (13 downto 0); -- available 3ns after COM_ADC_CLK-LH edge
         COM_ADC_CLK_N   : out std_logic;  --
         COM_ADC_CLK_P   : out std_logic;  -- 
         com_adc_dec_sdout  : out std_logic   -- decoder serial data output      
       );
  end component com_adc_dec;


  component receiver_8b10b is port
   (
    ena_rx          : in  std_logic;
    com_clk         : in  std_logic;
    baudrate_adj    : in std_logic_vector(1 downto 0);
    dec_8b10b_reset : out std_logic;
    rx_in           : in  std_logic;        -- serial data on the line
    rx_data_out     : out std_logic_vector (9 downto 0);  -- parallel out
    rx_data_valid   : out std_logic;        -- single clock length pulse
    dec_8b10b_valid : out std_logic;         -- delayed by one clock rx_data_valid to deal with the decoder latency
    rx_quiet        : out std_logic;         -- no signal on the line
    rx_stf          : out std_logic;         -- start of frame received
    rx_lh_nd        : out std_logic;        -- for debugging only
    rx_hl_nd        : out std_logic;        --
    rx_syncd_nd     : out std_logic;        --
    rx_bit_stb      : out std_logic
    );
  end component receiver_8b10b;

  component free_8b10b_dec is	port
   (
		RESET : in std_logic ;	-- Global asynchronous reset (AH) 
		RBYTECLK : in std_logic ;	-- Master synchronous receive byte clock
		AI, BI, CI, DI, EI, II : in std_logic ;
		FI, GI, HI, JI : in std_logic ; -- Encoded input (LS..MS)		
		KO : out std_logic ;	-- Control (K) character indicator (AH)
		HO, GO, FO, EO, DO, CO, BO, AO : out std_logic 	-- Decoded out (MS..LS)
	 );
  end component free_8b10b_dec;
  
  component rx_ctrl_8b10b is
   port(
        com_reset              :in  std_logic;   -- communication com_reset
        com_clk                :in  std_logic;   -- communication clock
        rx_quiet               :in  std_logic;   -- no signal, line is quiet
        ena_rx                 :in  std_logic;   --
        rx_done                :out std_logic;   -- handshake signal with ena_rx
        dec_8b10b_out          :in  std_logic_vector (7 downto 0); --
        dec_8b10b_valid        :in  std_logic;   --
        dec_8b10b_ko           :in  std_logic;   -- decoder output is comma
        STF_COMMA              :in  std_logic_vector (7 downto 0); -- start of frame comma
        EOF_COMMA              :in  std_logic_vector (7 downto 0);  -- end of frame comma        
        rx_fifo_almost_full    :in  std_logic;
        rx_fifo_wr_en          :out std_logic    -- pulse of one clock length
       );
  end component rx_ctrl_8b10b;

component mb_clk_sync IS
   PORT
   (
      fpga_clk_fbck : in  std_logic; -- feeded back 25 MHz clock
      fpga_clk_p    : out std_logic; --  mainboard fpga clock
      fpga_clk_n    : out std_logic; --     
      fpga_snc_p    : out std_logic; --  mainboard 1pps sync pulse of one fpga clock length
      fpga_snc_n    : out std_logic; -- 
      time_stamp    : out std_logic_vector(47 downto 0) -- 
    --time_stamp    : out natural range 0 to 2**48-1 -- 
    );
END component mb_clk_sync;

component half_duplex_ctrl_icm is
   port(
        com_reset          : in  std_logic;   -- communication com_reset
        com_clk            : in  std_logic;   -- communication clock
        tx_done            : in  std_logic;   --
        rx_done            : in  std_logic;   -- last reception finished 
        ena_tx             : out std_logic;   -- initiates sending of data or token
        ena_rx             : out std_logic;   -- initiates receiving of data or token
        rx_quiet           : in  std_logic;    -- rx, no signal on the line
        rx_stf             : in  std_logic   -- start of frame received 
       );
end component half_duplex_ctrl_icm;


  constant K28_0    : std_logic_vector(7 downto 0) := B"000_11100"; -- H...A
  constant K28_2    : std_logic_vector(7 downto 0) := B"010_11100"; -- H...A
  constant K28_4    : std_logic_vector(7 downto 0) := B"100_11100"; -- H...A
  constant K28_6    : std_logic_vector(7 downto 0) := B"110_11100"; -- H...A
  constant K28_7    : std_logic_vector(7 downto 0) := B"111_11100"; -- H...A
  
  constant K28_0M   : std_logic_vector(9 downto 0) := B"0010_111100"; -- j...a
  constant K28_0P   : std_logic_vector(9 downto 0) := B"1101_000011"; -- j...a
  
  constant COM_ADC_THR_mV    : natural range 0 to 255 := 2;  -- ADC data difference between consecutive ADC samples to detect signal edges
  constant COM_ADC_THRESHOLD : std_logic_vector :=  CONV_STD_LOGIC_VECTOR((COM_ADC_THR_mV * 1000)/ 61, 9); -- 14bit, Vref=1V, 0.061mV / digit


-- global signals

  signal sys_reset     : std_logic := '0'; -- synchronous power up reset
  signal sys_clk       : std_logic := '0'; -- by PLL
  signal com_reset     : std_logic := '0'; -- synchronous power up reset
  signal com_clk       : std_logic := '0'; -- by PLL, 8b10b communication clock domain
  signal mcu_reset     : std_logic := '0'; -- 
  signal mb_clkx2      : std_logic := '0'; -- main board clock
  signal mb_sync       : std_logic := '0'; -- main board sync pulse
  signal time_stamp    : std_logic_vector(47 downto 0);
  
-- half duplex arbiter signals
  signal ena_tx                 : std_logic; -- do transmit
  signal ena_rx                 : std_logic; -- do receive
  signal tx_done                : std_logic; -- transmission finished
  signal rx_done                : std_logic; -- reception finished
  signal rx_quiet               : std_logic; -- reception finished
  signal rx_stf                 : std_logic; -- start of frame received  
  
-- UART signals, rx/tx, FPGA view
  signal rx_uart_ena            : std_logic; -- from baudrate generator
  signal tx_uart_ena            : std_logic;
                             
  signal tx_uart_data_valid     : std_logic;
  signal tx_uart_data_in        : std_logic_vector (7 downto 0);
  signal tx_uart_ack            : std_logic;
  signal tx_uart_out            : std_logic;  -- connect to serial data output
                             
  signal rx_uart_in             : std_logic;  -- connect to serial data input
  signal rx_uart_data_out       : std_logic_vector (7 downto 0);
  signal rx_uart_data_valid     : std_logic;
                             
-- TxFIFO signals

  signal tx_fifo_wr_en          : std_logic;
  signal tx_fifo_rd_en          : std_logic;
  signal tx_fifo_dout           : std_logic_vector (7 downto 0);
  signal tx_fifo_full           : std_logic;
  signal tx_fifo_almost_full    : std_logic;
  signal tx_fifo_empty          : std_logic;
                                 
-- RxFIFO signals

  signal rx_fifo_wr_en          : std_logic;
  signal rx_fifo_rd_en          : std_logic;
  signal rx_fifo_dout           : std_logic_vector (7 downto 0);
  signal rx_fifo_din            : std_logic_vector (7 downto 0);
  signal rx_fifo_full           : std_logic;
  signal rx_fifo_prog_full      : std_logic;
  signal rx_fifo_empty          : std_logic;
  
-- 8b10b signals

  signal enc_8b10b_reset        : std_logic;
  signal enc_8b10b_in           : std_logic_vector (7 downto 0);
  signal enc_8b10b_ena          : std_logic;
  signal enc_comma_stb          : std_logic;
  signal tx_ena_8b10b           : std_logic;
  signal tx_data_in_8b10b       : std_logic_vector (9 downto 0);
  signal tx_run_8b10b           : std_logic;
  signal tx_done_8b10b          : std_logic;
  signal tx_ack_8b10b           : std_logic;
  signal tx_out_8b10b           : std_logic;
  signal tx_quiet_8b10b         : std_logic;
  signal baudrate_adj           : std_logic_vector(1 downto 0);
  
  signal rx_in_8b10b            : std_logic;
  signal rx_ena_8b10b           : std_logic;
  signal rx_data_out_8b10b      : std_logic_vector (9 downto 0);
  signal dec_8b10b_reset        : std_logic;
  signal dec_8b10b_out          : std_logic_vector (7 downto 0);
  signal rx_data_valid_8b10b    : std_logic;
  signal dec_8b10b_ko           : std_logic;
  signal dec_8b10b_valid        : std_logic;

  signal rx_lh                  : std_logic;        -- for debugging only
  signal rx_hl                  : std_logic;        --
  signal rx_syncd               : std_logic;        --
  signal rx_bit_stb            : std_logic;       --
  signal bit_length_ct_ena      : std_logic;       --
  signal com_thr_adj            : std_logic_vector(1 downto 0);
-- 8b10 encoder test
--  signal enc_test_d             : std_logic_vector (7 downto 0);
--  signal enc_test_ko            : std_logic;

  signal com_adc_dec_sdout         : std_logic; -- comm. ADC decoder output
   
 begin 
 
  -- clock, reset generation and baudrates ------------------------------------------------------- 
 
  pll_01_port_map: icm_pll_01 
    port map
   (
     qosc_clk      => QOSC_CLK,
     sys_clk       => sys_clk,   
     mb_clkx2      => mb_clkx2,
     com_clk       => com_clk, 
     not_com_clk   => open,
     com_reset     => com_reset,
     sys_reset     => sys_reset,
     mcu_reset     => mcu_reset
   );

   uart_baudrate_generator_port_map:
    uart_baudrate_generator
     generic map
     (
       CLK_FREQUENCY => 60_000_000,
       BAUD_RATE     =>  3_000_000  --115_200;--3_000_000;-- 256_000, 2_000_000
     )
     port map
     ( 
       reset    => sys_reset,
       clk      => sys_clk,
       tx_ena   => tx_uart_ena,
       rx_ena   => rx_uart_ena
     );

  cable_baudrate_generator_port_map:
   cable_baudrate_generator port map
    ( 
     reset        => com_reset,
     clk          => com_clk,
     baudrate_adj => baudrate_adj,
     tx_ena       => tx_ena_8b10b,
	 rx_ena       => rx_ena_8b10b
    );
 
  -- USB bridge to comm. DAC chain, see below -----------------------------------------------------------
 
 -- rx_uart_in_a <= tx_uart_out_a ;--

 --  rx_uart_in    <= STM_USART_TX;--FPGA_UART_TX;--USB_TXD ;  --   
 
        USB_CTSn      <= '1';--USB_RTSn and USB_PWRENn ;  -- dummy
        USB_RESETn    <= '1';
  
  uart_receiver_port_map:
   uart_receiver port map
   (
    reset          => sys_reset,
    clk            => sys_clk,
    rx_ena         => rx_uart_ena,
    rx_in          => rx_uart_in,
    rx_data_out    => rx_uart_data_out,
    rx_data_valid  => rx_uart_data_valid
    );
 
  
  tx_fifo_a_port_map:
   fifo_32kx8_dclk_pfull port map
   (
    rst           => sys_reset,
    wr_clk        => sys_clk,
    rd_clk        => com_clk,
    din           => rx_uart_data_out,
    wr_en         => rx_uart_data_valid,
    rd_en         => tx_fifo_rd_en,
    dout          => tx_fifo_dout,        
    empty         => tx_fifo_empty,
    prog_full     => tx_fifo_full,  
    full          => open        
  );
  
  tx_ctrl_8b10b_port_map:
   tx_ctrl_8b10b port map
  (
    com_reset       => ena_rx,--com_reset, 
    com_clk         => com_clk,
    ena_tx          => ena_tx,
    tx_done         => tx_done,     
    tx_fifo_empty   => tx_fifo_empty,   
    tx_fifo_rd_en   => tx_fifo_rd_en,   
    tx_fifo_dout    => tx_fifo_dout,
    STF_COMMA       => K28_0,       
    EOF_COMMA       => K28_7,       
    enc_8b10b_reset => enc_8b10b_reset,
    enc_8b10b_ena   => enc_8b10b_ena,
    enc_8b10b_in    => enc_8b10b_in,    
    enc_comma_stb   => enc_comma_stb,   
    tx_run_8b10b    => tx_run_8b10b,
    tx_ack_8b10b    => tx_ack_8b10b,
    tx_done_8b10b   => tx_done_8b10b 
   );
 
  enc_8b10b_port_map:
   enc_8b10b port map
    (
	reset     => com_reset,--enc_8b10b_reset,
	clk       => com_clk,
	ena       => enc_8b10b_ena,	
	data_in   => enc_8b10b_in,
    komma_in  => enc_comma_stb,
    data_out  => tx_data_in_8b10b
   );
 
  transmitter_8b10b_port_map:
   transmitter_8b10b port map
   (
     com_reset         => com_reset,
     com_clk           => com_clk,
     tx_run_8b10b      => tx_run_8b10b,
     tx_data_in_8b10b  => tx_data_in_8b10b,
     tx_ena_8b10b      => tx_ena_8b10b,
     tx_ack_8b10b      => tx_ack_8b10b,
     tx_done_8b10b     => tx_done_8b10b,
     tx_quiet_8b10b    => tx_quiet_8b10b,
	 tx_out_8b10b      => tx_out_8b10b
    );

  com_dac_enc_port_map:
   com_dac_enc port map
   (
    reset           => com_reset,
    clk             => com_clk,
	baudrate_adj    => baudrate_adj,
    sdin            => tx_out_8b10b,
    com_dac_quiet   => tx_quiet_8b10b,  
    com_dac_in      => COM_DAC_DB, -- upper 8 bits used only here
    com_dac_clock   => COM_DAC_CLOCK
   );
  

 -- comm. ADC to stamp chain, see below ------------------------------------------------------------
 
--      STAMP_DRXD    : out std_logic;  -- stamp debug uart 
--      STAMP_DTXD    : in  std_logic; --     
--      STAMP_RXD1    : out std_logic;  -- PB5, stamp uart
--      STAMP_TXD1    : in  std_logic; -- PB4,
 
  
  
 com_adc_dec_port_map:
  com_adc_dec port map
  (
     reset           => ena_tx,--com_reset,
     clk             => com_clk,           -- 60 MHz comm. clock
	 rx_ena          => '1',--rx_ena_8b10b,
	 baudrate_adj    => baudrate_adj,
     com_thr_adj     => com_thr_adj, 
     COM_ADC_CSBn    => COM_ADC_CSBn, 
     COM_ADC_SCLK    => COM_ADC_SCLK, 
     COM_ADC_SDIO    => COM_ADC_SDIO, 
     COM_ADC_D       => COM_ADC_D,    
     COM_ADC_CLK_N   => COM_ADC_CLK_N,
     COM_ADC_CLK_P   => COM_ADC_CLK_P,
     com_adc_dec_sdout   => com_adc_dec_sdout     -- decoder serial data output      
   );

   --rx_in_8b10b  <= tx_out_8b10b; -- for simulation
  rx_in_8b10b  <= com_adc_dec_sdout;

 receiver_8b10b_port_map:
   receiver_8b10b port map
   (
    ena_rx          => ena_rx,
    com_clk         => com_clk,
    baudrate_adj    => baudrate_adj,
    dec_8b10b_reset => dec_8b10b_reset,
    rx_in           => rx_in_8b10b,
    rx_data_out     => rx_data_out_8b10b,
    rx_data_valid   => rx_data_valid_8b10b,
    dec_8b10b_valid => dec_8b10b_valid,
    rx_quiet        => rx_quiet,
    rx_stf          => rx_stf,         -- start of frame received
    rx_lh_nd        => rx_lh,      -- 4 signals for debugging only      
    rx_hl_nd        => rx_hl,            
    rx_syncd_nd     => rx_syncd,         
    rx_bit_stb      => rx_bit_stb      
    );

 free_8b10b_dec_port_map:
   free_8b10b_dec port map
    (
	RESET    => dec_8b10b_reset, --com_reset, --
	RBYTECLK => com_clk,
	AI       => rx_data_out_8b10b(0),
    BI       => rx_data_out_8b10b(1),
    CI       => rx_data_out_8b10b(2),
    DI       => rx_data_out_8b10b(3),
    EI       => rx_data_out_8b10b(4),
    II       => rx_data_out_8b10b(5),
    FI       => rx_data_out_8b10b(6),
    GI       => rx_data_out_8b10b(7),
    HI       => rx_data_out_8b10b(8),
    JI       => rx_data_out_8b10b(9),  
   	KO       => dec_8b10b_ko,
    HO       => dec_8b10b_out(7),
    GO       => dec_8b10b_out(6),
    FO       => dec_8b10b_out(5),
    EO       => dec_8b10b_out(4),
    DO       => dec_8b10b_out(3),
    CO       => dec_8b10b_out(2),
    BO       => dec_8b10b_out(1),
    AO       => dec_8b10b_out(0) -- decoded out
	  );

   
   rx_ctrl_8b10b_port_map:
   rx_ctrl_8b10b port map
   (
    com_reset              => com_reset,
    com_clk                => com_clk,
    ena_rx                 => ena_rx,
    rx_done                => rx_done,
    rx_quiet               => rx_quiet,
    dec_8b10b_out          => dec_8b10b_out,  
    dec_8b10b_valid        => dec_8b10b_valid,
    dec_8b10b_ko           => dec_8b10b_ko,   
    STF_COMMA              => K28_0,
    EOF_COMMA              => K28_7,
    rx_fifo_almost_full    => rx_fifo_prog_full,
    rx_fifo_wr_en          => rx_fifo_wr_en
   );


rx_fifo_port_map:
   fifo_32kx8_dclk_pfull port map
   (
    rst       => com_reset,
    wr_clk    => com_clk,
    rd_clk    => sys_clk,
    din       => dec_8b10b_out,
    wr_en     => rx_fifo_wr_en,
    rd_en     => rx_fifo_rd_en,
    dout      => rx_fifo_dout, 
    empty     => rx_fifo_empty,
    prog_full => rx_fifo_prog_full,
    full      => open        
  );


   tx_uart_data_valid <= not rx_fifo_empty;   
  
  uart_transmitter_port_map:
   uart_transmitter port map
  (
    reset          => sys_reset,
    clk            => sys_clk,
    tx_data_valid  => tx_uart_data_valid,
    tx_data_in     => rx_fifo_dout,
    tx_ena         => tx_uart_ena,
    tx_ack         => rx_fifo_rd_en,
	tx_out         => tx_uart_out
   );

   tx_uart_data_valid <= not rx_fifo_empty;
   
 --STM_USART_RX     <= tx_uart_out;  -- 
   FPGA_UART_CTS    <= '0';          -- mDOT FPGA switches from USB- to ICM-UART
-- USB_RXD          <= tx_uart_out;  --
   USB_RXD          <= '1';    
 
 FPGA_CLK <= mb_clkx2; -- backlooped to MRCC clock pin as as FPGA_CLK_FBCK
 
mb_clk_sync_port_map:
   mb_clk_sync port map
   (
      fpga_clk_fbck =>  FPGA_CLK_FBCK,
      fpga_clk_p    =>  FPGA_CLK_P,
      fpga_clk_n    =>  FPGA_CLK_N,
      fpga_snc_p    =>  FPGA_SNC_P,
      fpga_snc_n    =>  FPGA_SNC_N,
      time_stamp    =>  time_stamp
    ); 
 
  half_duplex_ctrl_icm_port_map:  
    half_duplex_ctrl_icm port map
    (                     
      com_reset    =>  com_reset, 
      com_clk      =>  com_clk,   
      tx_done      =>  tx_done,   
      rx_done      =>  rx_done,   
      ena_tx       =>  ena_tx,    
      ena_rx       =>  ena_rx,
      rx_quiet     =>  rx_quiet,
      rx_stf       =>  rx_stf   
     );                    

  
  baudrate_adj(0) <= '0'; --TEST_IO_0 ; 
  baudrate_adj(1) <= '1'; --TEST_IO_3 ;

  com_thr_adj(0)  <= '0';--TEST_IO_0;
  com_thr_adj(1)  <= '0';--TEST_IO_3;
  
  TEST_IO_0       <= ena_rx; 
  TEST_IO_1       <= ena_tx;--rx_fifo_empty;--rx_fifo_wr_en;--com_adc_dec_sdout;
  TEST_IO_2       <= rx_done;--rx_fifo_rd_en;--'0';
  TEST_IO_3       <= tx_done;
 
  COM_DAC_SLEEP   <= '0';  -- 
           
  MBPWR_EN        <= '1'; --time_stamp(8);  -- MBPWR_EN now !!
  TERM_FPGA       <= '1';--time_stamp(16); --
   
  D00_MOSI        <= 'Z';  -- inout 
  D01_DIN         <= 'Z';  -- inout 
  D02             <= 'Z';  -- inout 
  D03             <= 'Z';  -- inout 
  FLASH_CSn       <= '1';  -- out   


-- 1.8V I/O level ----------------  
 CAL_TRG_OBUFDS_inst : OBUFDS
   generic map (
    IOSTANDARD => "DEFAULT", 
    SLEW => "fast") 
   port map 
    (
     O  => CAL_TRG_P, -- Diff_p output (connect directly to top-level port)
     OB => CAL_TRG_N, -- Diff_n output (connect directly to top-level port)
     I  => '0' -- Buffer input
    );

CAL_TIM_OBUFDS_inst : OBUFDS
   generic map (
    IOSTANDARD => "DEFAULT", 
    SLEW => "fast") 
   port map 
    (
     O  => CAL_TIM_P, -- Diff_p output (connect directly to top-level port)
     OB => CAL_TIM_N, -- Diff_n output (connect directly to top-level port)
     I  => '0' -- Buffer input
    );
-- 1.8V I/O level ----------------    
 FPGA_GPIO_0   <= 'Z';
 FPGA_GPIO_1   <= 'Z';
 
-- FPGA_UART_TX  <= ;
 FPGA_UART_RX  <= '1';
 FPGA_UART_CTS <= 'Z';
-- FPGA_UART_RTS <= ;

-- 3.3V I/O level ----------------
QOSC_SDA      <= 'Z';
QOSC_SCL      <= 'Z';
nLED_ENA      <= 'Z';
 
nRX_LED       <= '1'; 
nTX_LED       <= '1';
INTLK_0       <= '0'; 
INTLK_1       <= '0'; 
INTLK_2       <= '0'; 
INTLK_3       <= '0';              
MB_ID         <= 'Z';
ICM_ID        <= 'Z';    
MB_ADR0       <= 'Z';
MB_ADR1       <= 'Z';
MB_ADR2       <= 'Z';

--- MCU signals ------
MCU_nRST      <= not mcu_reset;
MCU_USART_CK  <= '0'; 
rx_uart_in    <= STM_USART_TX;
STM_USART_RX  <= tx_uart_out;
STM_USART_CTS <= 'Z';
--STM_USART_RTS <= 'Z';
STM_BOOT      <= '0'; -- ???

       
end arch_icm_top_009 ;
