-------------------------------------------------------------------------------
-- Design Name : icm_pll_01
-- File Name   : icm_pll_01.vhd
-- Device      : Spartan 7, XC7S25FTGB196-2
-- Function    : ICM pll design
-- Coder       : K.-H. Sulanke, DESY, 2019-12-20
-------------------------------------------------------------------------------
-- 2019-12-20; mcu_reset >1us added

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.numeric_std.all;

library unisim ;
use unisim.vcomponents.all ;

ENTITY icm_pll_01 IS
   PORT
   (
      qosc_clk    : IN  STD_LOGIC; --  20 MHz, local clock oscillator
      sys_clk     : OUT STD_LOGIC; --  60 MHz, system clock, made by pll_clkout2
      mb_clkx2    : OUT STD_LOGIC; --  50 MHz, mainboard clock x2, made by pll_clkout3  
      com_clk     : OUT STD_LOGIC; --  60 Mhz, communication clock made by pll_clkout4
      not_com_clk : OUT STD_LOGIC; --  60 Mhz, communication clock made by pll_clkout5, 180 phase shifted
      com_reset   : OUT STD_LOGIC; --  communication reset, synchronous to com_clk
      sys_reset   : OUT STD_LOGIC; --  synchronous to the system clock
      mcu_reset   : OUT STD_LOGIC  --  
    );
END icm_pll_01;

Architecture icm_pll_01_arch of icm_pll_01 is

  signal pll_rst       : std_logic := '0';    -- 
  signal pll_clkin1    : std_logic;  
  signal pll_clkfbin   : std_logic;    -- 
  signal pll_clkout2   : std_logic;    -- pll output,
  signal pll_clkout3   : std_logic;    -- pll output,
  signal pll_clkout4   : std_logic;    -- pll output,
  signal pll_clkout5   : std_logic;    -- pll output,
  
  signal pll_locked    : std_logic;    -- active high pll lock signal
  signal pll_reset_ct  : natural range 0 to 15 := 0; -- 
  signal reset_ct      : natural range 0 to 15 := 0; --  -- synchronous to sys_clk
  signal sys_reset_nd  : std_logic := '0';    -- 

  signal sys_clk_nd    : std_logic;    -- 
  signal mb_clkx4_nd   : std_logic;    -- 
  signal mb_clkx2_nd   : std_logic;    -- 
  signal com_clk_nd    : std_logic;    -- 
  signal not_com_clk_nd : std_logic;    --
   
  signal com_reset_nd  : std_logic := '0';    --
  signal com_reset_ct  : natural range 0 to 15 := 0;  -- synchronous to com_clk
  signal mcu_reset_nd  : std_logic := '0';    --
  signal mcu_reset_ct  : natural range 0 to 255 := 0;  -- synchronous to com_clk


begin
   
  pll_clkin_BUFG_inst : BUFG
   port map (
         O => pll_clkin1, -- 1-bit output: Clock buffer output
         I => qosc_clk -- 1-bit input: Clock buffer input
        );
     
 
  PLL_RESET_GEN: process(qosc_clk) 
   begin
     if rising_edge (qosc_clk) then
      if (pll_reset_ct /=15) then
        pll_reset_ct <= pll_reset_ct + 1;
      end if; --(pll_reset_ct /= 15)   
      if (pll_reset_ct = 14) then 
       pll_rst <= '1';
      else 
       pll_rst <= '0';
      end if; -- (pll_reset_ct = X"E") 
     end if; -- rising_edge (qosc_clk)
   end process PLL_RESET_GEN; 
     
   PLLE2_BASE_inst : PLLE2_BASE
   generic map (
     BANDWIDTH          => "OPTIMIZED", -- "HIGH", "LOW" or "OPTIMIZED"
     CLKFBOUT_MULT      =>  60, --PLL_MULT,   -- Multiplication factor for all output clocks, 800 MHz ...1866 MHz
     CLKFBOUT_PHASE     => 0.0, -- Phase shift (degrees) of all output clocks
     CLKIN1_PERIOD      => 50.0, --CLKIN_PERIOD, -- Clock period (ns) of input clock on CLKIN
     CLKOUT0_DIVIDE     =>  2,   -- Division factor for CLKOUT0 (1 to 128)
     CLKOUT0_DUTY_CYCLE => 0.5, -- Duty cycle for CLKOUT0 (0.01 to 0.99)
     CLKOUT0_PHASE      => 0.0, -- Phase shift (degrees) for CLKOUT0 (0.0 to 360.0)
     CLKOUT1_DIVIDE     =>  2,   -- Division factor for CLKOUT1 (1 to 128)
     CLKOUT1_DUTY_CYCLE => 0.5, -- Duty cycle for CLKOUT1 (0.01 to 0.99)
     CLKOUT1_PHASE      => 0.0, -- Phase shift (degrees) for CLKOUT1 (0.0 to 360.0)
     CLKOUT2_DIVIDE     => 20,   -- Division factor for CLKOUT2 (1 to 128)
     CLKOUT2_DUTY_CYCLE => 0.5, -- Duty cycle for CLKOUT2 (0.01 to 0.99)
     CLKOUT2_PHASE      => 0.0, -- Phase shift (degrees) for CLKOUT2 (0.0 to 360.0)
     CLKOUT3_DIVIDE     => 12, -- Division factor for CLKOUT3 (1 to 128)
     CLKOUT3_DUTY_CYCLE => 0.5, -- Duty cycle for CLKOUT3 (0.01 to 0.99)
     CLKOUT3_PHASE      => 0.0, -- Phase shift (degrees) for CLKOUT3 (0.0 to 360.0)
     CLKOUT4_DIVIDE     => 20, -- Division factor for CLKOUT4 (1 to 128)
     CLKOUT4_DUTY_CYCLE => 0.5, -- Duty cycle for CLKOUT4 (0.01 to 0.99)
     CLKOUT4_PHASE      => 0.0, -- Phase shift (degrees) for CLKOUT4 (0.0 to 360.0)
     CLKOUT5_DIVIDE     => 20, -- Division factor for CLKOUT5 (1 to 128)
     CLKOUT5_DUTY_CYCLE => 0.5, -- Duty cycle for CLKOUT5 (0.01 to 0.99)
     CLKOUT5_PHASE      => 180.0, -- Phase shift (degrees) for CLKOUT5 (0.0 to 360.0)
     DIVCLK_DIVIDE      => 1,--1,--5,--PLL_DIV, -- Division factor for all clocks (1 to 52)
     REF_JITTER1        => 0.010,  -- Input reference jitter (0.000 to 0.999 UI%)
     STARTUP_WAIT       => "TRUE")
     port map (
     CLKFBOUT           => pll_clkfbin, -- General output feedback signal
     CLKOUT0            => open, -- One of six general clock output signals
     CLKOUT1            => open, -- One of six general clock output signals
     CLKOUT2            => pll_clkout2, -- sys_clk
     CLKOUT3            => pll_clkout3, -- mb_clkx4
     CLKOUT4            => pll_clkout4, -- com_clk
     CLKOUT5            => pll_clkout5, -- not_com_clk
     LOCKED             => pll_locked, -- Active high PLL lock signal
     CLKFBIN            => pll_clkfbin, -- Clock feedback input
     CLKIN1             => pll_clkin1, -- Clock input
     PWRDWN             => '0',
     RST                => pll_rst --open -- Asynchronous PLL reset
   );
    

  SYNC_SYS_RESET: process ( sys_clk_nd)
    begin                           -- do power on reset, just once !
     if rising_edge(sys_clk_nd) then   
      --if (pll_locked ='1') then
       if reset_ct /= 15 then
        reset_ct <= reset_ct + 1;
       end if;
       if (reset_ct = 14) then
        sys_reset_nd <= '1';
       else
        sys_reset_nd <= '0';
       end if;
      ---end if; -- (pll_locked ='1') 
    end if; -- rising_edge(sys_clk_nd)
   end process SYNC_SYS_RESET;      


  SYNC_COM_RESET: process (com_clk_nd)
    begin                           -- do power on reset, just once !
     if rising_edge(com_clk_nd) then   
      --if (pll_locked ='1') then
       if com_reset_ct /= 15 then
        com_reset_ct <= com_reset_ct + 1;
       end if;
       if (com_reset_ct = 14) then
        com_reset_nd <= '1';
       else
        com_reset_nd <= '0';
       end if;
      --end if; -- (pll_locked ='1') 
    end if; -- rising_edge(com_clk_nd)
   end process SYNC_COM_RESET; 
   
   SYNC_MCU_RESET: process (com_clk_nd)
    begin                           -- do power on reset, just once !
     if rising_edge(com_clk_nd) then   
       if mcu_reset_ct /= 255 then
        mcu_reset_ct <= mcu_reset_ct + 1;
       end if;
       if (mcu_reset_ct = 1) then
        mcu_reset_nd <= '0';
       elsif (mcu_reset_ct = 10) then
        mcu_reset_nd <= '1';
       elsif (mcu_reset_ct = 200) then -- >1us needed
        mcu_reset_nd <= '0';
       end if;
      --end if; -- (pll_locked ='1') 
    end if; -- rising_edge(com_clk_nd)
   end process SYNC_MCU_RESET;     


  PLL_CLKOUT2_inst: BUFG port map ( I => pll_clkout2,  O => sys_clk_nd );      
  PLL_CLKOUT3_inst: BUFG port map ( I => pll_clkout3,  O => mb_clkx4_nd ); 
  PLL_CLKOUT4_inst: BUFG port map ( I => pll_clkout4,  O => com_clk_nd ); 
  PLL_CLKOUT5_inst: BUFG port map ( I => pll_clkout5,  O => not_com_clk_nd ); 
  
  MB_CLOCK_GEN: process (mb_clkx4_nd)
    begin                           -- do power on reset, just once !
     if rising_edge(mb_clkx4_nd) then   
      if (mb_clkx2_nd ='1') then
        mb_clkx2_nd <= '0';
      else
        mb_clkx2_nd <= '1';
      end if;  
    end if; -- rising_edge(mb_clkx4_nd)
   end process MB_CLOCK_GEN;      
  
  
  sys_clk    <= sys_clk_nd;
  mb_clkx2   <= mb_clkx2_nd;
  com_clk    <= com_clk_nd;
  sys_reset  <= sys_reset_nd;
  com_reset  <= com_reset_nd;
  mcu_reset  <= mcu_reset_nd;

END icm_pll_01_arch;
