-------------------------------------------------------------------------------
--
-- Title	: 8b/10b Encoder
-- Design	: 8-bit to 10-bit Encoder
-- Project	: 8000 - 8b10b_encdec
-- based on : 8b10b_enc.v by Chuck Benz, Hollis, NH
-------------------------------------------------------------------------------
--
-- File			: 8b10b_enc.vhd
-- Version		: 1.0
-- Generated	: 2019-12-04
-- By			: K.-H. Sulanke
--
-------------------------------------------------------------------------------
--
-- Description :
--	This module provides 8-bit to 10-bit encoding.
--	It accepts 8-bit parallel data input and generates 10-bit encoded data
--	output in accordance with the 8b/10b standard.  This coding method was
--	described in the 1983 IBM publication "A DC-Balanced, Partitioned-Block,
--	8B/10B Transmission Code" by A.X. Widmer and P.A. Franaszek and was granted
--	a U.S. Patent #4,486,739 in 1984 which has now expired.
--
library IEEE;
   use IEEE.std_logic_1164.all;
   use ieee.numeric_bit.all;
   use ieee.std_logic_arith.all;
library unisim ;
   use unisim.vcomponents.all ;

entity enc_8b10b is	
    port(
		reset    : in std_logic ;	                -- Global asynchronous reset (active high) 
		clk      : in std_logic ;	                -- synchronous  byte clock
		ena      : in std_logic ;                   -- clock enable
		data_in  : in std_logic_vector(7 downto 0);
		komma_in : in std_logic ;	                -- komma operator requested
        data_out : out std_logic_vector(9 downto 0) -- synchronous encoded out
	    );
end entity;

architecture enc_8b10b_arch of enc_8b10b is

-- Signals to tie things together
	signal disp_in  : std_logic ;	        -- 0 = neg disp; 1 = pos disp
	signal disp_out : std_logic ;	    -- 0 = neg disp; 1 = pos disp
	signal ai,bi,ci,di,ei,fi,gi,hi,ki : std_logic ;	-- 
	signal aeqb, ceqd : std_logic ;	--
	signal l22,l40,l04,l13,l31: std_logic ;	--
	signal ao,bo,co,do,eo,fo,go,ho,io,jo : std_logic ;	-- 
	signal pd1s4, nd1s4, pdos4, ndos4 : std_logic ;	-- 
	signal nd1s6, disp6, alt7 : std_logic ;	-- 
	signal ndos6, pd1s6, pdos6 : std_logic ;	-- 
	signal compls4, compls6 : std_logic ;	-- Figure 6 Signals 
    signal illegalk   : std_logic ;	--
    constant D0_0 : std_logic_vector := B"1101_000110";


begin


       ai <= data_in(0) ;
       bi <= data_in(1) ;
       ci <= data_in(2) ;
       di <= data_in(3) ;
       ei <= data_in(4) ;
       fi <= data_in(5) ;
       gi <= data_in(6) ;
       hi <= data_in(7) ;
       ki <= komma_in ;
      
       aeqb <= (ai AND bi) OR ( NOT ai AND  NOT bi) ;
       ceqd <= (ci AND di) OR ( NOT ci AND  NOT di) ;
       l22  <= (ai AND bi AND  NOT ci AND  NOT di) OR
               (ci AND di AND  NOT ai AND  NOT bi) OR
               (  NOT aeqb AND  NOT ceqd) ;
       l40  <= ai AND bi AND ci AND di ;
       l04  <=  NOT ai AND  NOT bi AND  NOT ci AND  NOT di ;
       l13  <= (  NOT aeqb AND  NOT ci AND  NOT di) OR
               (  NOT ceqd AND  NOT ai AND  NOT bi) ;
       l31  <= (  NOT aeqb AND ci AND di) OR
               (  NOT ceqd AND ai AND bi) ;
    
      -- The 5B/6B encoding
    
       ao <= ai ;
       bo <= (bi AND  NOT l40) OR l04 ;
       co <= l04 OR ci OR (ei AND di AND  NOT ci AND  NOT bi AND  NOT ai) ;
       do <= di AND  NOT  (ai AND bi AND ci) ;
       eo <= (ei OR l13) AND  NOT  (ei AND di AND  NOT ci AND  NOT bi AND  NOT ai) ;
       io <= (l22 AND  NOT ei) OR
            (ei AND  NOT di AND  NOT ci AND  NOT (ai AND bi)) OR  -- D16, D17, D18
            (ei AND l40) OR
            (ki AND ei AND di AND ci AND  NOT bi AND  NOT ai) OR -- K.28
            (ei AND  NOT di AND ci AND  NOT bi AND  NOT ai) ;
    
      -- pds16 indicates cases where d-1 is assumed + to get our encoded value
       pd1s6 <= (ei AND di AND  NOT ci AND  NOT bi AND  NOT ai) OR ( NOT ei AND  NOT l22 AND  NOT l31) ;
      -- nds16 indicates cases where d-1 is assumed - to get our encoded value
       nd1s6 <= ki OR (ei AND  NOT l22 AND  NOT l13) OR ( NOT ei AND  NOT di AND ci AND bi AND ai) ;
    
      -- ndos6 is pds16 cases where d-1 is + yields - disp out - all of them
       ndos6 <= pd1s6 ;
      -- pdos6 is nds16 cases where d-1 is - yields + disp out - all but one
       pdos6 <= ki OR (ei AND  NOT l22 AND  NOT l13) ;
    
    
      -- some Dx.7 and all Kx.7 cases result in run length of 5 case unless
      -- an alternate coding is used (referred to as Dx.A7, normal is Dx.P7)
      -- specifically, D11, D13, D14, D17, D18, D19.
    --   alt7 <= fi AND gi AND hi AND (ki OR 
    --			      (disp_in ? ( NOT ei AND di AND l31) : (ei AND  NOT di AND l13))) ;
    ----   alt7 <= fi AND gi AND hi AND (ki OR (disp_in and NOT ei AND di AND l31) OR (NOT disp_in and ei AND  NOT di AND l13)) ;
           alt7 <= (fi AND gi AND hi AND (ki OR ( NOT ei AND di AND l31)) and disp_in) or 
                   (fi AND gi AND hi AND (ki OR (ei AND  NOT di AND l13)) and not disp_in);
       
       fo <= fi AND  NOT  alt7 ;
       go <= gi OR ( NOT fi AND  NOT gi AND  NOT hi) ;
       ho <= hi ;
       jo <= ( NOT hi AND (gi XOR fi)) OR alt7 ;
    
      -- nd1s4 is cases where d-1 is assumed - to get our encoded value
       nd1s4 <= fi AND gi ;
      -- pd1s4 is cases where d-1 is assumed + to get our encoded value
       pd1s4 <= ( NOT fi AND  NOT gi) OR (ki AND ((fi AND  NOT gi) OR ( NOT fi AND gi))) ;
    
      -- ndos4 is pd1s4 cases where d-1 is + yields - disp out - just some
       ndos4 <= ( NOT fi AND  NOT gi) ;
      -- pdos4 is nd1s4 cases where d-1 is - yields + disp out 
       pdos4 <= fi AND gi AND hi ;
    
      -- only legal K codes are K28.0->.7, K23/27/29/30.7
      --	K28.0->7 is ei<=di<=ci<=1,bi<=ai<=0
      --	K23 is 10111
      --	K27 is 11011
      --	K29 is 11101
      --	K30 is 11110 - so K23/27/29/30 are ei AND l31
       illegalk <= ki AND 
              (ai OR bi OR  NOT ci OR  NOT di OR  NOT ei) AND -- not K28.0->7
              ( NOT fi OR  NOT gi OR  NOT hi OR  NOT ei OR  NOT l31) ; -- not K23/27/29/30.7
    
      -- now determine whether to do the complementing
      -- complement if prev disp is - and pd1s6 is set, or + and nd1s6 is set
       compls6 <= (pd1s6 AND  NOT disp_in) OR (nd1s6 AND disp_in) ;
   
      -- disparity out of 5b6b is disp in with pdso6 and ndso6
      -- pds16 indicates cases where d-1 is assumed + to get our encoded value
      -- ndos6 is cases where d-1 is + yields - disp out
      -- nds16 indicates cases where d-1 is assumed - to get our encoded value
      -- pdos6 is cases where d-1 is - yields + disp out
      -- disp toggles in all ndis16 cases, and all but that 1 nds16 case
    
       disp6    <= disp_in XOR (ndos6 OR pdos6) ;
    
       compls4  <= (pd1s4 AND  NOT disp6) OR (nd1s4 AND disp6) ;
       disp_out <= disp6 XOR (ndos4 OR pdos4) ;

 ENCODE_8b10b: process(clk)
  begin
   if rising_edge(clk) then
    if reset = '1' then
     data_out <= D0_0;
     disp_in  <= '0';
    elsif ena = '1' then
--       if disp_out = '1' then
--        disp_in <= not disp_in;
--       end if;
       disp_in  <= disp_out;  
       data_out <= ((jo XOR compls4), (ho XOR compls4),
                    (go XOR compls4), (fo XOR compls4),
                    (io XOR compls6), (eo XOR compls6),
                    (do XOR compls6), (co XOR compls6),
                    (bo XOR compls6), (ao XOR compls6)) ;                  
    end if; -- reset = '1'
   end if; --rising_edge(clk)
  end process ENCODE_8b10b;                 

				
end enc_8b10b_arch;

