-------------------------------------------------------
-- Design Name : com_adc_dec 
-- File Name   : com_adc_dec.vhd
-- ADC chip    : AD9649
-- Function    : serial data recovery from com_adc output signal 
-- Coder       : K.-H. Sulanke, DESY
-- Date        : 2019-10-17
-- Revision    : 12
-------------------------------------------------------
-- to be used for IceCube gen1 encoding style, decoding the trailing edge only
-- ADC resolution is 2V / 14 bit = 0,122 mV / bit -> ~ 8bit / 1mV 
-- with 3.5 km filter box + 3MBd, 8b10b+ASK, sine pk-pk  is ~20 mV
-- with 2 km cable , 2MBd, 8b10b+ASK, falling edge pk-pk  is ~30 mV, used threshold was 20 mV

library ieee;
    use ieee.std_logic_1164.all;
    use ieee.std_logic_unsigned.all;
    use ieee.numeric_bit.all;
    use ieee.std_logic_arith.all;
Library UNISIM;
    use UNISIM.vcomponents.all;

entity com_adc_dec is
   port(
         reset           : in  std_logic;
         clk             : in  std_logic; -- 60 MHz comm. clock
		 rx_ena          : in  std_logic;
		 baudrate_adj    : in std_logic_vector(1 downto 0);
         com_thr_adj     : in  std_logic_vector (1 downto 0); -- set the comm. threshold
         COM_ADC_CSBn    : out std_logic;   --
         COM_ADC_SCLK    : out std_logic;   --
         COM_ADC_SDIO    : inout std_logic; --
         COM_ADC_D       : in  std_logic_vector (13 downto 0); -- available 3ns after COM_ADC_CLK-LH edge
         COM_ADC_CLK_N   : out std_logic;  --
         COM_ADC_CLK_P   : out std_logic;  -- 
         com_adc_dec_sdout : out std_logic  -- decoder serial data output
--       du              : in  natural range 0 to 2**14-1    -- typical transistion / fall time is 13 mV / ADC tick (33ns)        
       );
end entity;

architecture com_adc_dec_arch of com_adc_dec is

  constant ADC_CLK          : integer := 30_000_000;
  constant BAUD_RATE        : integer :=  2_000_000;  -- 3_000_000
  constant MAX_PULSE_LENGTH : natural:= ADC_CLK * 2 / BAUD_RATE; -- each '1' pulse corresponds to ... rx_ena pulses or ADC clocks
 
  constant mV : natural := 8;
  constant dU : natural := 3 * mV; -- per 33ns (30MHz ADC-clock) 
  
  constant  N : natural := 2;
  type   array_Nx14 is array (0 to N-1) of natural range 0 to 2**14 -1; 
  signal com_adc_pipe      : array_Nx14 := ( others => 0);
 
  signal com_adc_clock     : std_logic := '0';
  signal com_adc_thr       : natural range 0 to 2**14-1;
  signal com_adc_dn_sum    : natural range 0 to 2**14-1;
   
  signal com_adc_above_thr     : std_logic; 
  signal com_adc_dec_sdout_nd  : std_logic;
  
  signal adc_sum_ena       : std_logic;
  signal up_going_edge     : std_logic;
  signal down_going_edge   : std_logic;
 
  signal   pulse_length_ct  : natural range 0 to MAX_PULSE_LENGTH;

  type dec_state_type is (DEC_IDLE, FALLING_EDGE);
	signal dec_state: dec_state_type := DEC_IDLE;
  
 begin      
 
  set_com_thr: process (clk)
   begin
   if rising_edge(clk) then
     if reset = '1' then
      com_adc_thr <=   10 * mV; --
     elsif rx_ena = '1' then 
        case com_thr_adj is
         when B"00"   => com_adc_thr <=    10 * mV; -- 
         when B"01"   => com_adc_thr <=    15 * mV; -- 
         when B"10"   => com_adc_thr <=    20 * mV; -- 
         when B"11"   => com_adc_thr <=    25 * mV; -- 
         when others  => com_adc_thr <=    10 * mV; --                
        end case;
      end if; -- reset = '1' 
     end if; -- rising_edge(clk)   
   end process set_com_thr; 

   dec_sm: process (clk)
	begin
	  if (rising_edge(clk)) then
	   if (reset = '1') then
          dec_state    <= DEC_IDLE;
       elsif (rx_ena ='1') then
       
		case (dec_state) is
			  
         when DEC_IDLE =>
          adc_sum_ena <= '1';
          if    com_adc_dn_sum > com_adc_thr then
           dec_state   <= FALLING_EDGE;
           adc_sum_ena <= '0';
          end if; 
         
         when FALLING_EDGE =>	
          if (up_going_edge = '1') then
           dec_state   <= DEC_IDLE;
          end if;
                   
		 when others =>
		   dec_state    <= DEC_IDLE;
				  
		end case;
	   end if; --(reset ='1')			
	 end if; -- rising_edge(clk)
    end process dec_sm; 		


  com_adc_decoder: process (clk)
    begin
     if (rising_edge(clk)) then
      if (reset ='1') then
       com_adc_clock     <= '0';
       com_adc_above_thr <= '0';
      elsif rx_ena = '1' then
       com_adc_above_thr <= '0'; -- to get single pulses
       com_adc_clock <= not com_adc_clock;       
       if com_adc_clock = '1' then    -- get data after LH edge
         com_adc_pipe(0) <= conv_integer(COM_ADC_D);
         com_adc_pipe(1) <= com_adc_pipe(0);         
         if (com_adc_pipe(0) > com_adc_pipe(1) + dU) then
          up_going_edge <= '1';
         else
          up_going_edge <= '0';
         end if;
         if (com_adc_pipe(1) > com_adc_pipe(0) + dU) then
          down_going_edge <= '1';
         else
          down_going_edge <= '0';
         end if;
         if (adc_sum_ena = '1') then         
          if (com_adc_pipe(1) > com_adc_pipe(0) + dU ) then
           com_adc_dn_sum <= com_adc_dn_sum + (com_adc_pipe(1) - com_adc_pipe(0));
          else
           com_adc_dn_sum <= 0;
          end if; 
         else            
           com_adc_dn_sum <= 0;
         end if;   -- (adc_sum_ena = '1')
         if (com_adc_dn_sum > com_adc_thr)  then
           com_adc_above_thr <= '1';
           com_adc_dn_sum <= 0;
         end if; --        
       end if; -- com_adc_clock = '1'     
      end if; -- (reset = '1') 
     end if; --  (rising_edge(clk))
   end process com_adc_decoder; 

   pulse_keeper: process (clk)
    begin
     if (rising_edge(clk)) then
      if (reset ='1') then
       com_adc_dec_sdout_nd <= '0';
      elsif rx_ena = '1' then
       if com_adc_above_thr = '1' then -- single pulse
        pulse_length_ct      <=  0;
        com_adc_dec_sdout_nd <= '1';
       elsif  pulse_length_ct < MAX_PULSE_LENGTH-1 then
        pulse_length_ct  <= pulse_length_ct  + 1;
       else
        com_adc_dec_sdout_nd <= '0';
       end if; -- com_adc_above_thr = '1'
      end if; --(reset ='1')
     end if; --(rising_edge(clk))
    end process pulse_keeper; 

   com_adc_dec_sdout <=  com_adc_dec_sdout_nd;    
       
  COM_ADC_CLK_inst : OBUFDS
   generic map (
     IOSTANDARD => "DEFAULT")
   port map (
      O  => COM_ADC_CLK_P,   -- Diff_p output (connect directly to top-level port)
      OB => COM_ADC_CLK_N,   -- Diff_n output (connect directly to top-level port)
      I  => com_adc_clock    -- Buffer input 
   );


   COM_ADC_CSBn    <= '1';   -- serial interface disabled
   COM_ADC_SCLK    <= '0';   -- offset binary
   COM_ADC_SDIO    <= '0';   -- normal operation

 end architecture com_adc_dec_arch;

