-----------------------------------------able_baudrate_generator 
-- File Name   : cable_baudrate_generator.vhd
-- Function    : generate tx enables from the system clock
--               with 10 fold oversampling, according to the baudrate_adj setting
-- Coder       : K.-H. Sulanke, DESY
-- Date        : 2019-10-06
------------------------------------------------------------------------------------

library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_unsigned.all;

entity cable_baudrate_generator is
  port ( 
         reset          : in  std_logic;
         clk            : in  std_logic; -- 60 MHz
         baudrate_adj   : in std_logic_vector(1 downto 0); -- ICM jumpers on (TEST_IO_3, TEST_IO_0)
         tx_ena         : out std_logic;
		 rx_ena         : out std_logic
         );
end entity cable_baudrate_generator;

architecture cable_baudrate_generator_arch of cable_baudrate_generator is
 
  signal tx_baud_div    : natural range 0 to 127;
  signal rx_baud_div    : natural range 0 to   7;
  
  signal tx_clock_count : natural range 0 to 127;  
  signal rx_clock_count : natural range 0 to   7; 
   
  begin
    
  set_baudrate: process (clk)
   begin
   if rising_edge(clk) then
     if reset = '1' then
       tx_baud_div <= 60 - 1; rx_baud_div <= 6 - 1;
     else 
       if tx_clock_count = tx_baud_div then  -- for synchronized changing of the baudrate    
        case baudrate_adj is
         when B"00"   => tx_baud_div <= 60 - 1; rx_baud_div <= 6 - 1; --  1000_000 baud @ 60MHz clk
         when B"01"   => tx_baud_div <= 40 - 1; rx_baud_div <= 4 - 1; --  1500_000 baud @ 60MHz clk
         when B"10"   => tx_baud_div <= 30 - 1; rx_baud_div <= 3 - 1; --  2000_000 baud @ 60MHz clk
         when B"11"   => tx_baud_div <= 20 - 1; rx_baud_div <= 2 - 1; --  3000_000 baud @ 60MHz clk
                                                            
         when others => tx_baud_div <= 60 - 1; rx_baud_div <= 6 - 1;              
        end case;
       end if; -- tx_ena = '1'
      end if; -- reset = '1' 
     end if; -- rising_edge(clk)   
   end process set_baudrate; 
        
  make_tx_ena: process (clk, reset)
  begin
    if rising_edge(clk) then
      if reset = '1' then
        tx_clock_count <= 0;
        tx_ena         <= '0';
      elsif tx_clock_count = tx_baud_div then
        tx_ena         <= '1';
        tx_clock_count <= 0;
      else
        tx_clock_count <= tx_clock_count +1;
        tx_ena         <= '0';
      end if; 
    end if;
  end process make_tx_ena;

  make_rx_ena: process (clk, reset)
  begin
    if rising_edge(clk) then
      if reset = '1' then
        rx_clock_count <= 0;
        rx_ena         <= '0';
      elsif rx_clock_count = rx_baud_div then
        rx_ena         <= '1';
        rx_clock_count <= 0;
      else
        rx_clock_count <= rx_clock_count +1;
        rx_ena         <= '0';
      end if; 
    end if;
  end process make_rx_ena;


end architecture cable_baudrate_generator_arch;
